README.md

Telegram Multi-Tenant Bot System

Complete Installation & Configuration Guide

---

📋 Table of Contents

1. System Overview
2. Prerequisites
3. Quick Installation
4. Database Configuration
5. Environment Variables
6. Server Deployment
7. Domain & SSL Setup
8. Payment Gateway Setup
9. Telegram Bot Configuration
10. Maintenance & Backup
11. Troubleshooting

---

🎯 System Overview

Telegram Multi-Tenant Bot System is an enterprise-grade platform for managing multiple Telegram bots with complete activity tracking, dual key management, payment integration, and more.

Key Features

· 👑 Owner Panel: Manage all admins, view all users, monitor all activities
· 🤖 Admin Panel: Independent bot control, key management, payment gateway
· 👥 User Interface: Simple purchase flow, key delivery, APK downloads
· 🔒 Security: Role-based access, encrypted data, audit logs
· 📊 Analytics: Complete activity tracking, revenue reports, user insights

---

📋 Prerequisites

Server Requirements

Component Minimum Recommended
CPU 2 cores 4+ cores
RAM 4 GB 8+ GB
Storage 20 GB SSD 50+ GB SSD
OS Ubuntu 20.04 Ubuntu 22.04
Domain Valid domain with SSL -

Software Requirements

· Docker 20.10+
· Docker Compose 2.0+
· Git
· Python 3.11+
· Node.js 18+
· Nginx

---

⚡ Quick Installation

1. Clone Repository

```bash
git clone https://github.com/yourusername/telegram-multi-tenant-bot.git
cd telegram-multi-tenant-bot
```

2. Configure Environment

```bash
# Copy example environment file
cp .env.example .env

# Edit with your configuration
nano .env
```

3. Run Deployment Script

```bash
# Make script executable
chmod +x scripts/deploy.sh

# Run deployment
./scripts/deploy.sh
```

4. Create Initial Admin

```bash
# Run the create admin script
python scripts/create_admin.py
```

5. Set Webhooks

```bash
# Set webhook for all bots
python scripts/set_webhook.py set-all --url https://your-domain.com
```

---

🗄️ Database Configuration

Master Database Credentials

Parameter Value
Database Name telegram_bot_master
Username telegram_admin
Password [YOUR_STRONG_PASSWORD]
Host localhost (or postgres in Docker)
Port 5432

Connection Strings

```bash
# Local connection (for development)
postgresql://telegram_admin:YOUR_PASSWORD@localhost:5432/telegram_bot_master

# Docker connection (for containerized environment)
postgresql://telegram_admin:YOUR_PASSWORD@postgres:5432/telegram_bot_master

# Remote connection (with SSL for production)
postgresql://telegram_admin:YOUR_PASSWORD@your-domain.com:5432/telegram_bot_master?sslmode=require
```

Database Setup Commands

Access PostgreSQL

```bash
# Via Docker
docker exec -it telegram_bot_postgres psql -U telegram_admin -d telegram_bot_master

# Via local installation
psql -U telegram_admin -d telegram_bot_master -h localhost
```

Useful PostgreSQL Commands

```sql
-- List all databases
\l

-- Connect to database
\c telegram_bot_master

-- List all tables
\dt

-- Describe table structure
\d table_name

-- Show database size
SELECT pg_database_size('telegram_bot_master')/1024/1024 AS size_mb;

-- Show all connections
SELECT * FROM pg_stat_activity;

-- Kill a connection
SELECT pg_terminate_backend(pid) FROM pg_stat_activity WHERE pid = 12345;
```

Database Migrations

```bash
# Run migrations
docker exec -it telegram_bot_backend alembic upgrade head

# Create new migration
docker exec -it telegram_bot_backend alembic revision --autogenerate -m "description"

# Rollback migration
docker exec -it telegram_bot_backend alembic downgrade -1

# View migration history
docker exec -it telegram_bot_backend alembic history
```

Backup & Restore

```bash
# Create backup
docker exec -it telegram_bot_postgres pg_dump -U telegram_admin telegram_bot_master > backup_$(date +%Y%m%d).sql

# Create compressed backup
docker exec -it telegram_bot_postgres pg_dump -U telegram_admin telegram_bot_master | gzip > backup_$(date +%Y%m%d).sql.gz

# Restore backup
cat backup.sql | docker exec -i telegram_bot_postgres psql -U telegram_admin telegram_bot_master

# Restore compressed backup
gunzip -c backup.sql.gz | docker exec -i telegram_bot_postgres psql -U telegram_admin telegram_bot_master
```

Tenant Databases

Each admin gets their own isolated database:

```sql
-- Format: tenant_[RESERVE_ID]
-- Examples:
-- tenant_game_001
-- tenant_movie_002
-- tenant_vpn_003

-- List all tenant databases
SELECT datname FROM pg_database WHERE datname LIKE 'tenant_%';

-- Get size of all tenant databases
SELECT 
    datname,
    pg_database_size(datname)/1024/1024 AS size_mb
FROM pg_database 
WHERE datname LIKE 'tenant_%'
ORDER BY size_mb DESC;
```

---

🔐 Environment Variables

Core Settings

```bash
# .env file - Copy and modify these values

# ============================================
# APPLICATION SETTINGS
# ============================================

APP_NAME="Telegram Multi-Tenant Bot System"
ENVIRONMENT=production              # development, staging, production
DEBUG=false                          # Set to true only in development
SECRET_KEY=your-super-secret-key-here-change-in-production
JWT_SECRET_KEY=your-jwt-secret-key-here-change-in-production
JWT_ALGORITHM=HS256
JWT_ACCESS_TOKEN_EXPIRE_MINUTES=30
JWT_REFRESH_TOKEN_EXPIRE_DAYS=7

# ============================================
# DATABASE SETTINGS
# ============================================

# PostgreSQL
DB_HOST=postgres                      # Use 'localhost' for local setup
DB_PORT=5432
DB_NAME=telegram_bot_master
DB_USER=telegram_admin
DB_PASSWORD=YourStrongPassword123!    # CHANGE THIS!
DATABASE_URL=postgresql://${DB_USER}:${DB_PASSWORD}@${DB_HOST}:${DB_PORT}/${DB_NAME}
DATABASE_POOL_SIZE=20
DATABASE_MAX_OVERFLOW=10
DATABASE_POOL_TIMEOUT=30

# ============================================
# REDIS CACHE SETTINGS
# ============================================

REDIS_HOST=redis                       # Use 'localhost' for local setup
REDIS_PORT=6379
REDIS_PASSWORD=YourRedisPassword123!   # CHANGE THIS!
REDIS_URL=redis://:${REDIS_PASSWORD}@${REDIS_HOST}:${REDIS_PORT}/0
REDIS_MAX_CONNECTIONS=10
REDIS_SOCKET_TIMEOUT=5

# ============================================
# RABBITMQ SETTINGS
# ============================================

RABBITMQ_HOST=rabbitmq                 # Use 'localhost' for local setup
RABBITMQ_PORT=5672
RABBITMQ_USER=telegram_bot
RABBITMQ_PASSWORD=YourRabbitPassword123! # CHANGE THIS!
RABBITMQ_VHOST=/
RABBITMQ_URL=amqp://${RABBITMQ_USER}:${RABBITMQ_PASSWORD}@${RABBITMQ_HOST}:${RABBITMQ_PORT}${RABBITMQ_VHOST}

# ============================================
# CELERY SETTINGS
# ============================================

CELERY_BROKER_URL=${RABBITMQ_URL}
CELERY_RESULT_BACKEND=${REDIS_URL}
CELERY_TASK_SERIALIZER=json
CELERY_RESULT_SERIALIZER=json
CELERY_ACCEPT_CONTENT=['json']
CELERY_TIMEZONE=UTC
CELERY_TASK_TRACK_STARTED=true
CELERY_TASK_TIME_LIMIT=1800            # 30 minutes
CELERY_TASK_SOFT_TIME_LIMIT=1500       # 25 minutes

# ============================================
# MINIO STORAGE (for APK files)
# ============================================

MINIO_ENDPOINT=minio:9000               # Use 'localhost:9000' for local setup
MINIO_ACCESS_KEY=minioadmin
MINIO_SECRET_KEY=YourMinioPassword123!  # CHANGE THIS!
MINIO_SECURE=false                       # Set to true for production with SSL
MINIO_BUCKET_APK=apk-files
MINIO_BUCKET_BACKUPS=backups
MINIO_REGION=us-east-1

# ============================================
# TELEGRAM SETTINGS
# ============================================

# Get these from https://my.telegram.org
TELEGRAM_API_ID=123456                  # Your API ID
TELEGRAM_API_HASH=your-telegram-api-hash

# Get this from @BotFather on Telegram
TELEGRAM_BOT_TOKEN=1234567890:ABCdefGHIjklMNOpqrsTUVwxyz

# Webhook settings
TELEGRAM_WEBHOOK_URL=https://your-domain.com
TELEGRAM_WEBHOOK_SECRET=your-webhook-secret-here  # Random string
TELEGRAM_MAX_CONNECTIONS=100

# ============================================
# PAYMENT GATEWAYS
# ============================================

# Default gateway
DEFAULT_PAYMENT_GATEWAY=razorpay        # razorpay, stripe, phonepe, paytm

# Razorpay (https://dashboard.razorpay.com)
RAZORPAY_KEY_ID=rzp_live_xxxxxxxxxxxx
RAZORPAY_KEY_SECRET=xxxxxxxxxxxxxxxxxxxxxxxx
RAZORPAY_WEBHOOK_SECRET=xxxxxxxxxxxxxxxx

# Stripe (https://dashboard.stripe.com)
STRIPE_PUBLIC_KEY=pk_live_xxxxxxxxxxxx
STRIPE_SECRET_KEY=sk_live_xxxxxxxxxxxx
STRIPE_WEBHOOK_SECRET=whsec_xxxxxxxxxxxx

# PhonePe (https://developer.phonepe.com)
PHONEPE_MERCHANT_ID=your_merchant_id
PHONEPE_SALT_KEY=your_salt_key
PHONEPE_SALT_INDEX=1
PHONEPE_ENV=PROD                          # UAT for testing, PROD for production

# PayPal
PAYPAL_CLIENT_ID=your_paypal_client_id
PAYPAL_CLIENT_SECRET=your_paypal_secret
PAYPAL_MODE=sandbox                       # sandbox or live

# Paytm
PAYTM_MERCHANT_ID=your_merchant_id
PAYTM_MERCHANT_KEY=your_merchant_key
PAYTM_WEBSITE=DEFAULT
PAYTM_INDUSTRY_TYPE=Retail

# ============================================
# EMAIL SETTINGS (for notifications)
# ============================================

SMTP_HOST=smtp.gmail.com
SMTP_PORT=587
SMTP_USER=your-email@gmail.com
SMTP_PASSWORD=your-app-password            # Gmail app password
SMTP_USE_TLS=true
EMAIL_FROM=noreply@your-domain.com
EMAIL_FROM_NAME="Telegram Bot System"
EMAIL_REPLY_TO=support@your-domain.com

# ============================================
# MONITORING & LOGGING
# ============================================

# Prometheus
PROMETHEUS_ENABLED=true
PROMETHEUS_PORT=9090

# Sentry (for error tracking)
SENTRY_DSN=https://xxxxxxxxxxxx@sentry.io/xxxxxx
SENTRY_ENVIRONMENT=production
SENTRY_TRACES_SAMPLE_RATE=1.0

# Logging
LOG_LEVEL=INFO                             # DEBUG, INFO, WARNING, ERROR
LOG_FORMAT=json                             # json or text
LOG_FILE=/app/logs/app.log
LOG_RETENTION_DAYS=30
LOG_MAX_SIZE=100MB

# ============================================
# BACKUP SETTINGS
# ============================================

BACKUP_ENABLED=true
BACKUP_SCHEDULE="0 2 * * *"                 # Daily at 2 AM (cron format)
BACKUP_RETENTION_DAYS=30
BACKUP_PATH=/app/backups
BACKUP_COMPRESSION=gzip                     # gzip or none

# AWS S3 Backup (optional)
AWS_ACCESS_KEY_ID=your_aws_access_key
AWS_SECRET_ACCESS_KEY=your_aws_secret_key
AWS_BUCKET_NAME=your-backup-bucket
AWS_REGION=us-east-1

# ============================================
# RATE LIMITING
# ============================================

RATE_LIMIT_ENABLED=true
RATE_LIMIT_DEFAULT=100/minute
RATE_LIMIT_AUTH=10/minute
RATE_LIMIT_BROADCAST=5/minute
RATE_LIMIT_KEY_GEN=10/minute
RATE_LIMIT_WEBHOOK=1000/minute

# ============================================
# FILE UPLOAD SETTINGS
# ============================================

MAX_UPLOAD_SIZE=104857600                    # 100MB in bytes
ALLOWED_EXTENSIONS=.apk,.zip,.csv,.xlsx,.jpg,.png
UPLOAD_PATH=/app/uploads
TEMP_PATH=/app/tmp

# ============================================
# KEY GENERATION SETTINGS
# ============================================

KEY_MIN_LENGTH=8
KEY_MAX_LENGTH=50
KEY_DEFAULT_PATTERN="{PREFIX}-{YYYYMMDD}-{RANDOM8}"
KEY_EXCLUDE_SIMILAR=true
KEY_CHECKSUM_ENABLED=false
KEY_BATCH_SIZE=1000
KEY_LOW_STOCK_ALERT=100
KEY_AUTO_GENERATE=true
KEY_AUTO_GEN_THRESHOLD=50

# ============================================
# ACTIVITY TRACKING
# ============================================

TRACK_USER_ACTIVITY=true
TRACK_ADMIN_ACTIVITY=true
ACTIVITY_RETENTION_DAYS=90
SESSION_RETENTION_DAYS=30
TRACK_IP_ADDRESS=true
TRACK_USER_AGENT=true

# ============================================
# BROADCAST SETTINGS
# ============================================

BROADCAST_BATCH_SIZE=100
BROADCAST_MAX_RETRIES=3
BROADCAST_RETRY_DELAY=60                     # seconds
BROADCAST_RATE_LIMIT=30/second
BROADCAST_MAX_MESSAGE_LENGTH=4096

# ============================================
# CACHE SETTINGS
# ============================================

CACHE_TTL_DEFAULT=300                        # 5 minutes
CACHE_TTL_USER=3600                          # 1 hour
CACHE_TTL_KEY=600                            # 10 minutes
CACHE_TTL_PLAN=3600                          # 1 hour
CACHE_TTL_ADMIN=300                          # 5 minutes
CACHE_TTL_STATS=600                          # 10 minutes

# ============================================
# SECURITY & CORS
# ============================================

CORS_ORIGINS=https://your-domain.com,https://admin.your-domain.com
CORS_ALLOW_CREDENTIALS=true
CORS_ALLOW_METHODS=GET,POST,PUT,DELETE,OPTIONS
CORS_ALLOW_HEADERS=Content-Type,Authorization,Accept

SESSION_TIMEOUT=3600                         # 1 hour
SESSION_COOKIE_SECURE=true
SESSION_COOKIE_HTTPONLY=true
SESSION_COOKIE_SAMESITE=Lax

# ============================================
# DOMAIN SETTINGS
# ============================================

DOMAIN_NAME=your-domain.com
ADMIN_DOMAIN=admin.your-domain.com
API_DOMAIN=api.your-domain.com
OWNER_DOMAIN=owner.your-domain.com

# ============================================
# FRONTEND SETTINGS
# ============================================

REACT_APP_API_URL=https://${API_DOMAIN}
REACT_APP_OWNER_URL=https://${OWNER_DOMAIN}
REACT_APP_ADMIN_URL=https://${ADMIN_DOMAIN}

# ============================================
# DASHBOARD SETTINGS
# ============================================

DASHBOARD_REFRESH_INTERVAL=30                 # seconds
DASHBOARD_DEFAULT_PERIOD=week                 # day, week, month, year
CHARTS_ENABLED=true
EXPORT_ENABLED=true

# ============================================
# API SETTINGS
# ============================================

API_VERSION=v1
API_PREFIX=/api/${API_VERSION}
API_DOCS_ENABLED=true                          # Set to false in production
API_DOCS_URL=/api/docs

# ============================================
# WEBHOOK SETTINGS
# ============================================

WEBHOOK_BASE_PATH=/webhook
WEBHOOK_PAYMENT_PATH=/payments/webhook
WEBHOOK_TELEGRAM_PATH=/telegram
WEBHOOK_SECRET_HEADER=X-Webhook-Secret

# ============================================
# NOTIFICATION SETTINGS
# ============================================

NOTIFICATION_TELEGRAM_ENABLED=true
NOTIFICATION_EMAIL_ENABLED=true
NOTIFICATION_ADMIN_ON_PAYMENT=true
NOTIFICATION_ADMIN_ON_LOW_KEYS=true
NOTIFICATION_OWNER_ON_ADMIN_CREATE=true
NOTIFICATION_OWNER_ON_EXPIRY=true
```

Generate Secure Keys

```bash
# Generate SECRET_KEY (32 bytes random)
python -c "import secrets; print(secrets.token_urlsafe(32))"

# Generate JWT_SECRET_KEY
python -c "import secrets; print(secrets.token_urlsafe(32))"

# Generate database password (16 chars alphanumeric)
python -c "import secrets; import string; chars = string.ascii_letters + string.digits; print(''.join(secrets.choice(chars) for _ in range(16)))"

# Generate Redis password
python -c "import secrets; print(secrets.token_urlsafe(16))"

# Generate RabbitMQ password
python -c "import secrets; print(secrets.token_urlsafe(16))"

# Generate MinIO password
python -c "import secrets; print(secrets.token_urlsafe(16))"

# Generate webhook secret
python -c "import secrets; print(secrets.token_urlsafe(32))"

# One-liner to generate all keys
echo "SECRET_KEY=$(python -c 'import secrets; print(secrets.token_urlsafe(32))')"
echo "JWT_SECRET_KEY=$(python -c 'import secrets; print(secrets.token_urlsafe(32))')"
echo "DB_PASSWORD=$(python -c 'import secrets; import string; chars = string.ascii_letters + string.digits; print(\"\".join(secrets.choice(chars) for _ in range(16)))')"
echo "REDIS_PASSWORD=$(python -c 'import secrets; print(secrets.token_urlsafe(16))')"
echo "RABBITMQ_PASSWORD=$(python -c 'import secrets; print(secrets.token_urlsafe(16))')"
echo "MINIO_SECRET_KEY=$(python -c 'import secrets; print(secrets.token_urlsafe(16))')"
```

---

🚀 Server Deployment

Deploy with Docker Compose

1. Prepare Server

```bash
# Update system
sudo apt update && sudo apt upgrade -y

# Install Docker
curl -fsSL https://get.docker.com -o get-docker.sh
sudo sh get-docker.sh

# Install Docker Compose
sudo curl -L "https://github.com/docker/compose/releases/latest/download/docker-compose-$(uname -s)-$(uname -m)" -o /usr/local/bin/docker-compose
sudo chmod +x /usr/local/bin/docker-compose

# Add user to docker group
sudo usermod -aG docker $USER
newgrp docker
```

2. Configure Firewall

```bash
# Install UFW if not present
sudo apt install -y ufw

# Configure rules
sudo ufw default deny incoming
sudo ufw default allow outgoing
sudo ufw allow 22/tcp        # SSH
sudo ufw allow 80/tcp        # HTTP
sudo ufw allow 443/tcp       # HTTPS
sudo ufw allow 5432/tcp      # PostgreSQL (internal only)
sudo ufw allow 6379/tcp      # Redis (internal only)
sudo ufw allow 5672/tcp      # RabbitMQ (internal only)

# Enable firewall
sudo ufw --force enable

# Check status
sudo ufw status verbose
```

3. Deploy Application

```bash
# Clone repository
git clone https://github.com/yourusername/telegram-multi-tenant-bot.git
cd telegram-multi-tenant-bot

# Configure environment
cp .env.example .env
nano .env  # Edit with your values

# Build and start containers
docker-compose up -d --build

# Check logs
docker-compose logs -f

# Check container status
docker-compose ps
```

4. Initialize Database

```bash
# Run migrations
docker exec -it telegram_bot_backend alembic upgrade head

# Create initial admin
docker exec -it telegram_bot_backend python scripts/create_admin.py

# Seed recharge plans
docker exec -it telegram_bot_postgres psql -U telegram_admin -d telegram_bot_master -f database/seeds/recharge_plans.sql
```

5. Set Webhooks

```bash
# Set webhook for all bots
docker exec -it telegram_bot_backend python scripts/set_webhook.py set-all --url https://your-domain.com

# Verify webhook
docker exec -it telegram_bot_backend python scripts/set_webhook.py get --token YOUR_BOT_TOKEN
```

Useful Docker Commands

```bash
# View all containers
docker ps -a

# View logs
docker logs telegram_bot_backend --tail 100 -f

# Execute command in container
docker exec -it telegram_bot_backend bash

# Restart service
docker-compose restart backend

# Stop all services
docker-compose down

# Stop and remove volumes (WARNING: deletes data)
docker-compose down -v

# Rebuild and restart
docker-compose up -d --build

# Scale backend (for load balancing)
docker-compose up -d --scale backend=3 backend

# View resource usage
docker stats

# Clean up unused resources
docker system prune -a
docker volume prune
```

---

🌐 Domain & SSL Setup

Configure DNS Records

Add these records at your domain provider:

Type Name Value
A your-domain.com YOUR_SERVER_IP
A api.your-domain.com YOUR_SERVER_IP
A admin.your-domain.com YOUR_SERVER_IP
A owner.your-domain.com YOUR_SERVER_IP

Install SSL with Let's Encrypt

```bash
# Install certbot
sudo apt install -y certbot python3-certbot-nginx

# Obtain certificate
sudo certbot --nginx -d your-domain.com -d api.your-domain.com -d admin.your-domain.com -d owner.your-domain.com

# Test auto-renewal
sudo certbot renew --dry-run

# Check certificate status
sudo certbot certificates
```

Nginx Configuration

```nginx
# /etc/nginx/sites-available/telegram-bot

# Upstream definitions
upstream backend {
    server 127.0.0.1:8000;
    keepalive 32;
}

# Owner Dashboard
server {
    listen 443 ssl http2;
    server_name owner.your-domain.com;

    ssl_certificate /etc/letsencrypt/live/your-domain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/your-domain.com/privkey.pem;
    
    ssl_protocols TLSv1.2 TLSv1.3;
    ssl_ciphers HIGH:!aNULL:!MD5;
    
    location / {
        proxy_pass http://localhost:3001;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}

# Admin Dashboard
server {
    listen 443 ssl http2;
    server_name admin.your-domain.com;

    ssl_certificate /etc/letsencrypt/live/your-domain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/your-domain.com/privkey.pem;
    
    location / {
        proxy_pass http://localhost:3002;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}

# API Backend
server {
    listen 443 ssl http2;
    server_name api.your-domain.com;

    ssl_certificate /etc/letsencrypt/live/your-domain.com/fullchain.pem;
    ssl_certificate_key /etc/letsencrypt/live/your-domain.com/privkey.pem;
    
    location / {
        proxy_pass http://backend;
        proxy_http_version 1.1;
        proxy_set_header Upgrade $http_upgrade;
        proxy_set_header Connection 'upgrade';
        proxy_set_header Host $host;
        proxy_set_header X-Real-IP $remote_addr;
        proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto $scheme;
        proxy_cache_bypass $http_upgrade;
    }
}

# HTTP redirect
server {
    listen 80;
    server_name your-domain.com api.your-domain.com admin.your-domain.com owner.your-domain.com;
    return 301 https://$server_name$request_uri;
}
```

```bash
# Enable site
sudo ln -s /etc/nginx/sites-available/telegram-bot /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl reload nginx
```

---

💳 Payment Gateway Setup

Razorpay

```bash
1. Go to https://dashboard.razorpay.com
2. Sign up/Login to your account
3. Navigate to Settings → API Keys
4. Generate new API key
5. Copy Key ID and Key Secret
6. Add to .env:
   RAZORPAY_KEY_ID=rzp_live_xxxxxxxxxxxx
   RAZORPAY_KEY_SECRET=xxxxxxxxxxxxxxxx
   RAZORPAY_WEBHOOK_SECRET=xxxxxxxxxxxxxxxx
7. Configure webhook: https://api.your-domain.com/api/v1/payments/webhook/razorpay
8. Select events: payment.captured, payment.failed, refund.processed
```

Stripe

```bash
1. Go to https://dashboard.stripe.com
2. Navigate to Developers → API Keys
3. Copy Publishable key and Secret key
4. Add to .env:
   STRIPE_PUBLIC_KEY=pk_live_xxxxxxxxxxxx
   STRIPE_SECRET_KEY=sk_live_xxxxxxxxxxxx
   STRIPE_WEBHOOK_SECRET=whsec_xxxxxxxxxxxx
5. Configure webhook: https://api.your-domain.com/api/v1/payments/webhook/stripe
6. Select events: payment_intent.succeeded, payment_intent.payment_failed
```

PhonePe

```bash
1. Register at https://developer.phonepe.com
2. Get Merchant ID and Salt Key from dashboard
3. Add to .env:
   PHONEPE_MERCHANT_ID=your_merchant_id
   PHONEPE_SALT_KEY=your_salt_key
   PHONEPE_SALT_INDEX=1
   PHONEPE_ENV=PROD
4. Configure callback URL: https://api.your-domain.com/api/v1/payments/webhook/phonepe
```

Test Cards for Development

Razorpay Test Cards

Card Number Expiry CVV Description
4111 1111 1111 1111 Any future Any Successful payment
4000 0000 0000 0002 Any future Any Failed payment

Stripe Test Cards

Card Number Description
4242 4242 4242 4242 Successful payment
4000 0000 0000 0002 Declined payment
4000 0000 0000 9995 Insufficient funds

---

🤖 Telegram Bot Configuration

Create Bot with BotFather

```bash
1. Open Telegram and search for @BotFather
2. Send /newbot command
3. Choose a name for your bot (e.g., "Gaming Keys Bot")
4. Choose a username (e.g., "gaming_keys_bot")
5. Copy the bot token (format: 1234567890:ABCdefGHIjklMNOpqrsTUVwxyz)
6. Add to .env: TELEGRAM_BOT_TOKEN=your_bot_token
```

Configure Bot Settings

```bash
# Set bot commands (send to @BotFather)
/setcommands

# Add these commands:
start - 🚀 Start the bot
plans - 📋 View available plans
buy - 🛒 Purchase keys
my_keys - 🔑 View your keys
download - 📱 Download APK
extend - ⏰ Extend key
help - ❓ Get help
contact - 📧 Contact support

# Set bot description
/setdescription - Describe what your bot does

# Set bot profile picture
/setuserpic - Upload bot logo (512x512px recommended)

# Set bot privacy (send to @BotFather)
/setprivacy - Set to DISABLED to see all messages
```

Webhook Configuration

```bash
# Set webhook manually
curl -F "url=https://api.your-domain.com/webhook/YOUR_BOT_TOKEN" \
     -F "max_connections=100" \
     -F "allowed_updates=['message','callback_query']" \
     https://api.telegram.org/botYOUR_BOT_TOKEN/setWebhook

# Check webhook status
curl https://api.telegram.org/botYOUR_BOT_TOKEN/getWebhookInfo

# Expected response:
{
  "ok": true,
  "result": {
    "url": "https://api.your-domain.com/webhook/123456:ABCdef",
    "has_custom_certificate": false,
    "pending_update_count": 0,
    "max_connections": 100
  }
}

# Delete webhook
curl https://api.telegram.org/botYOUR_BOT_TOKEN/deleteWebhook
```

---

💾 Maintenance & Backup

Automated Backup Script

Create /usr/local/bin/backup-telegram-bot.sh:

```bash
#!/bin/bash
# Telegram Bot System Backup Script

# Configuration
BACKUP_DIR="/backups/telegram-bot"
DATE=$(date +%Y%m%d_%H%M%S)
RETENTION_DAYS=30
LOG_FILE="/var/log/telegram-bot-backup.log"
PROJECT_DIR="/path/to/telegram-multi-tenant-bot"

# Load environment
source $PROJECT_DIR/.env

# Logging function
log() {
    echo "[$(date '+%Y-%m-%d %H:%M:%S')] $1" | tee -a $LOG_FILE
}

# Start backup
log "=========================================="
log "Starting Telegram Bot System Backup"
log "=========================================="

# Create backup directories
mkdir -p $BACKUP_DIR/{database,uploads,config,logs}

# 1. Backup PostgreSQL databases
log "Backing up PostgreSQL databases..."

# Master database
log "  - Backing up master database: $DB_NAME"
docker exec telegram_bot_postgres pg_dump -U $DB_USER $DB_NAME | gzip > $BACKUP_DIR/database/master_$DATE.sql.gz
MASTER_SIZE=$(du -h $BACKUP_DIR/database/master_$DATE.sql.gz | cut -f1)
log "    Size: $MASTER_SIZE"

# Tenant databases
log "  - Backing up tenant databases..."
docker exec telegram_bot_postgres psql -U $DB_USER -t -c "SELECT datname FROM pg_database WHERE datname LIKE 'tenant_%'" | while read dbname; do
    if [ ! -z "$dbname" ]; then
        dbname=$(echo $dbname | xargs)
        log "    - Backing up $dbname"
        docker exec telegram_bot_postgres pg_dump -U $DB_USER $dbname | gzip > $BACKUP_DIR/database/${dbname}_$DATE.sql.gz
    fi
done

# 2. Backup Redis
log "Backing up Redis..."
docker exec telegram_bot_redis redis-cli -a $REDIS_PASSWORD SAVE
docker cp telegram_bot_redis:/data/dump.rdb $BACKUP_DIR/database/redis_$DATE.rdb
REDIS_SIZE=$(du -h $BACKUP_DIR/database/redis_$DATE.rdb | cut -f1)
log "  Size: $REDIS_SIZE"

# 3. Backup uploaded files
log "Backing up uploads..."
tar -czf $BACKUP_DIR/uploads/uploads_$DATE.tar.gz -C $PROJECT_DIR/backend/uploads . 2>/dev/null
UPLOADS_SIZE=$(du -h $BACKUP_DIR/uploads/uploads_$DATE.tar.gz | cut -f1)
log "  Size: $UPLOADS_SIZE"

# 4. Backup configuration
log "Backing up configuration..."
cp $PROJECT_DIR/.env $BACKUP_DIR/config/env_$DATE.bak
cp $PROJECT_DIR/docker-compose.yml $BACKUP_DIR/config/docker-compose_$DATE.yml
CONFIG_SIZE=$(du -h $BACKUP_DIR/config/env_$DATE.bak | cut -f1)
log "  Size: $CONFIG_SIZE"

# 5. Backup logs
log "Backing up logs..."
tar -czf $BACKUP_DIR/logs/logs_$DATE.tar.gz -C $PROJECT_DIR/backend/logs . 2>/dev/null
LOGS_SIZE=$(du -h $BACKUP_DIR/logs/logs_$DATE.tar.gz | cut -f1)
log "  Size: $LOGS_SIZE"

# 6. Upload to S3 (optional)
if [ ! -z "$AWS_ACCESS_KEY_ID" ]; then
    log "Uploading to S3..."
    aws s3 sync $BACKUP_DIR/ s3://$AWS_BUCKET_NAME/telegram-bot/$DATE/ --quiet
    log "  Upload completed"
fi

# 7. Clean old backups
log "Cleaning backups older than $RETENTION_DAYS days..."
find $BACKUP_DIR -type f -mtime +$RETENTION_DAYS -delete
find $BACKUP_DIR -type d -empty -delete

# 8. Calculate total size
TOTAL_SIZE=$(du -sh $BACKUP_DIR | cut -f1)

# 9. Send notification
log "Sending notification..."
curl -s -X POST "https://api.telegram.org/bot$TELEGRAM_BOT_TOKEN/sendMessage" \
     -d "chat_id=$OWNER_TELEGRAM_ID" \
     -d "text=✅ Backup completed successfully!%0A%0A📅 Date: $DATE%0A📦 Total Size: $TOTAL_SIZE%0A💾 Location: $BACKUP_DIR" \
     -d "parse_mode=HTML" > /dev/null

# Summary
log "=========================================="
log "Backup Summary:"
log "  Date: $DATE"
log "  Master DB: $MASTER_SIZE"
log "  Redis: $REDIS_SIZE"
log "  Uploads: $UPLOADS_SIZE"
log "  Config: $CONFIG_SIZE"
log "  Logs: $LOGS_SIZE"
log "  Total Size: $TOTAL_SIZE"
log "  Location: $BACKUP_DIR"
log "=========================================="
log "Backup completed successfully!"
log "=========================================="
```

```bash
# Make script executable
chmod +x /usr/local/bin/backup-telegram-bot.sh

# Test backup
/usr/local/bin/backup-telegram-bot.sh
```

Cron Jobs Setup

```bash
# Edit crontab
crontab -e

# Add these lines:

# Daily backup at 2 AM
0 2 * * * /usr/local/bin/backup-telegram-bot.sh >> /var/log/telegram-bot-backup.log 2>&1

# Weekly backup verification (Sunday at 3 AM)
0 3 * * 0 /usr/local/bin/verify-backup.sh >> /var/log/verify-backup.log 2>&1

# Monitor disk space (every hour)
0 * * * * /usr/local/bin/check-disk.sh >> /var/log/disk-check.log 2>&1

# Restart services weekly (Sunday at 4 AM)
0 4 * * 0 cd /path/to/telegram-multi-tenant-bot && docker-compose restart

# Clean old logs (daily at 1 AM)
0 1 * * * find /var/log -name "*.log" -mtime +30 -delete

# Database vacuum (weekly on Sunday at 5 AM)
0 5 * * 0 docker exec telegram_bot_postgres vacuumdb -U telegram_admin -d telegram_bot_master -v -z
```

Restore Script

Create /usr/local/bin/restore-telegram-bot.sh:

```bash
#!/bin/bash
# Telegram Bot System Restore Script

# Configuration
BACKUP_FILE=$1
BACKUP_DIR="/backups/telegram-bot"
PROJECT_DIR="/path/to/telegram-multi-tenant-bot"

if [ -z "$BACKUP_FILE" ]; then
    echo "Usage: $0 <backup-file>"
    echo "Available backups:"
    ls -lh $BACKUP_DIR/database/*.sql.gz | awk '{print "  " $9 " (" $5 ")"}'
    exit 1
fi

echo "=========================================="
echo "Telegram Bot System Restore"
echo "=========================================="
echo "Backup file: $BACKUP_FILE"
echo ""

# Extract date from filename
BACKUP_DATE=$(basename $BACKUP_FILE | grep -oP '\d{8}_\d{6}')

if [ -z "$BACKUP_DATE" ]; then
    echo "Error: Could not extract date from filename"
    exit 1
fi

echo "Backup date: $BACKUP_DATE"
echo ""

# Confirm restore
read -p "WARNING: This will overwrite current data. Continue? (y/N) " -n 1 -r
echo
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    echo "Restore cancelled."
    exit 0
fi

# Stop services
echo "Stopping services..."
cd $PROJECT_DIR
docker-compose stop backend celery_worker

# Restore master database
echo "Restoring master database..."
gunzip -c $BACKUP_FILE | docker exec -i telegram_bot_postgres psql -U telegram_admin telegram_bot_master

# Restore tenant databases
echo "Restoring tenant databases..."
for tenant_backup in $BACKUP_DIR/database/tenant_*_$BACKUP_DATE.sql.gz; do
    if [ -f "$tenant_backup" ]; then
        dbname=$(basename $tenant_backup | sed "s/_$BACKUP_DATE.sql.gz//")
        echo "  - Restoring $dbname"
        gunzip -c $tenant_backup | docker exec -i telegram_bot_postgres psql -U telegram_admin $dbname
    fi
done

# Restore Redis
echo "Restoring Redis..."
if [ -f "$BACKUP_DIR/database/redis_$BACKUP_DATE.rdb" ]; then
    docker cp $BACKUP_DIR/database/redis_$BACKUP_DATE.rdb telegram_bot_redis:/data/dump.rdb
    docker-compose restart redis
fi

# Restore uploads
echo "Restoring uploads..."
if [ -f "$BACKUP_DIR/uploads/uploads_$BACKUP_DATE.tar.gz" ]; then
    tar -xzf $BACKUP_DIR/uploads/uploads_$BACKUP_DATE.tar.gz -C $PROJECT_DIR/backend/uploads
fi

# Restore configuration
echo "Restoring configuration..."
if [ -f "$BACKUP_DIR/config/env_$BACKUP_DATE.bak" ]; then
    cp $BACKUP_DIR/config/env_$BACKUP_DATE.bak $PROJECT_DIR/.env
fi

# Start services
echo "Starting services..."
docker-compose start backend celery_worker

echo ""
echo "Restore completed successfully!"
echo "=========================================="
```

```bash
# Make script executable
chmod +x /usr/local/bin/restore-telegram-bot.sh
```

---

🔧 Troubleshooting

Common Issues & Solutions

1. Database Connection Errors

```bash
# Check if PostgreSQL is running
docker ps | grep postgres

# Check logs
docker logs telegram_bot_postgres --tail 50

# Test connection
docker exec -it telegram_bot_postgres psql -U telegram_admin -d telegram_bot_master -c "SELECT 1"

# Check max connections
docker exec -it telegram_bot_postgres psql -U telegram_admin -c "SHOW max_connections;"

# Check current connections
docker exec -it telegram_bot_postgres psql -U telegram_admin -c "SELECT count(*) FROM pg_stat_activity;"

# Fix: Increase max connections in postgresql.conf
docker exec -it telegram_bot_postgres bash -c "echo 'max_connections = 200' >> /var/lib/postgresql/data/postgresql.conf"
docker-compose restart postgres
```

2. Redis Connection Issues

```bash
# Check if Redis is running
docker ps | grep redis

# Test connection
docker exec -it telegram_bot_redis redis-cli -a $REDIS_PASSWORD ping

# Monitor Redis
docker exec -it telegram_bot_redis redis-cli -a $REDIS_PASSWORD monitor

# Check memory usage
docker exec -it telegram_bot_redis redis-cli -a $REDIS_PASSWORD info memory

# Clear Redis cache (if needed)
docker exec -it telegram_bot_redis redis-cli -a $REDIS_PASSWORD FLUSHALL

# Fix: Increase maxmemory in redis.conf
docker exec -it telegram_bot_redis redis-cli -a $REDIS_PASSWORD CONFIG SET maxmemory 2gb
```

3. RabbitMQ Issues

```bash
# Check if RabbitMQ is running
docker ps | grep rabbitmq

# Check queues
docker exec -it telegram_bot_rabbitmq rabbitmqctl list_queues

# Check connections
docker exec -it telegram_bot_rabbitmq rabbitmqctl list_connections

# Check consumers
docker exec -it telegram_bot_rabbitmq rabbitmqctl list_consumers

# Restart RabbitMQ
docker-compose restart rabbitmq

# Fix: Reset RabbitMQ (WARNING: deletes all queues)
docker exec -it telegram_bot_rabbitmq rabbitmqctl stop_app
docker exec -it telegram_bot_rabbitmq rabbitmqctl reset
docker exec -it telegram_bot_rabbitmq rabbitmqctl start_app
```

4. Celery Worker Issues

```bash
# Check worker status
docker exec -it telegram_bot_celery celery -A api.tasks.celery_app status

# Check queue length
docker exec -it telegram_bot_rabbitmq rabbitmqctl list_queues

# View worker logs
docker logs telegram_bot_celery --tail 100 -f

# Check registered tasks
docker exec -it telegram_bot_celery celery -A api.tasks.celery_app inspect registered

# Restart workers
docker-compose restart celery_worker celery_beat

# Fix: Purge all tasks (WARNING: deletes pending tasks)
docker exec -it telegram_bot_celery celery -A api.tasks.celery_app purge -f
```

5. Webhook Issues

```bash
# Check webhook info
curl https://api.telegram.org/bot$TELEGRAM_BOT_TOKEN/getWebhookInfo

# Test webhook manually
curl -X POST https://api.your-domain.com/webhook/$TELEGRAM_BOT_TOKEN \
  -H "Content-Type: application/json" \
  -d '{"update_id": 123456789, "message": {"message_id": 1, "from": {"id": 123456789, "is_bot": false, "first_name": "Test"}, "chat": {"id": 123456789, "first_name": "Test", "type": "private"}, "date": 1609459200, "text": "/start"}}'

# Check SSL certificate
openssl s_client -connect api.your-domain.com:443 -servername api.your-domain.com

# Fix: Reset webhook
curl -F "url=https://api.your-domain.com/webhook/$TELEGRAM_BOT_TOKEN" https://api.telegram.org/bot$TELEGRAM_BOT_TOKEN/setWebhook
```

6. Nginx Issues

```bash
# Test configuration
sudo nginx -t

# Check error logs
sudo tail -f /var/log/nginx/error.log

# Check access logs
sudo tail -f /var/log/nginx/access.log

# Check if port 80/443 are listening
sudo netstat -tlnp | grep -E ':(80|443)'

# Reload nginx
sudo systemctl reload nginx

# Restart nginx
sudo systemctl restart nginx
```

7. Docker Issues

```bash
# Check all containers
docker ps -a

# Check container logs
docker logs container_name --tail 100

# Check disk usage
docker system df

# Check container resource usage
docker stats

# Clean up unused resources
docker system prune -a
docker volume prune
docker network prune

# Fix: Restart Docker daemon
sudo systemctl restart docker
```

8. Permission Issues

```bash
# Fix file permissions
sudo chown -R $USER:$USER /path/to/telegram-multi-tenant-bot
sudo chmod -R 755 /path/to/telegram-multi-tenant-bot

# Fix Docker socket permission
sudo usermod -aG docker $USER
newgrp docker

# Fix upload directory permissions
sudo chmod -R 777 /path/to/telegram-multi-tenant-bot/backend/uploads
```

9. Memory Issues

```bash
# Check memory usage
free -h
docker stats --no-stream

# Check swap usage
swapon --show

# Limit container memory in docker-compose.yml
services:
  backend:
    deploy:
      resources:
        limits:
          memory: 1G

# Clear system cache
sudo sync && sudo echo 3 > /proc/sys/vm/drop_caches
```

10. Performance Issues

```bash
# Check slow queries
docker exec -it telegram_bot_postgres psql -U telegram_admin -c "SELECT query, calls, total_time, mean_time FROM pg_stat_statements ORDER BY mean_time DESC LIMIT 10;"

# Add indexes for slow queries
docker exec -it telegram_bot_postgres psql -U telegram_admin -d telegram_bot_master -c "CREATE INDEX IF NOT EXISTS idx_activity_created ON user_activity_log(created_at);"

# Enable query logging
docker exec -it telegram_bot_postgres bash -c "echo 'log_min_duration_statement = 1000' >> /var/lib/postgresql/data/postgresql.conf"
docker-compose restart postgres
```

---

📊 Monitoring & Alerts

Check System Health

```bash
# System health endpoint
curl https://api.your-domain.com/health

# Expected response:
{
  "status": "healthy",
  "timestamp": "2024-01-20T10:30:00Z",
  "checks": {
    "database": "up",
    "redis": "up",
    "rabbitmq": "up"
  }
}
```

View Metrics in Grafana

```bash
# Access Grafana
URL: https://your-domain.com:3000
Username: admin
Password: admin (change on first login)

# Import dashboard
1. Login to Grafana
2. Click + → Import
3. Upload dashboard.json from monitoring/grafana/dashboards/
4. Select Prometheus datasource
5. Click Import
```

---

📞 Support

Contact Information

· Email: support@your-domain.com
· Telegram: @support_bot
· Documentation: https://docs.your-domain.com
· GitHub Issues: https://github.com/yourusername/telegram-multi-tenant-bot/issues

Useful Commands Quick Reference

```bash
# View all logs
docker-compose logs -f

# Restart all services
docker-compose restart

# Update application
git pull
docker-compose up -d --build

# Run database migrations
docker exec -it telegram_bot_backend alembic upgrade head

# Create backup
/usr/local/bin/backup-telegram-bot.sh

# Check status
docker-compose ps
```

---

✅ Final Checklist

Before going live, ensure:

· All environment variables are set correctly
· Database is properly configured and backed up
· SSL certificates are installed and valid
· Payment gateways are configured and tested
· Telegram webhook is set and responding
· Firewall rules are configured
· Backup system is working
· Monitoring is set up
· Initial admin account is created
· Test transaction works end-to-end
· Domain DNS records are propagated
· Email notifications are working
· Rate limiting is enabled
· Error tracking (Sentry) is configured

---

Deployment Complete! 🎉

Your Telegram Multi-Tenant Bot System is now ready to use.

Access your dashboards:

· Owner Panel: https://owner.your-domain.com
· Admin Panel: https://admin.your-domain.com
· API Docs: https://api.your-domain.com/api/docs
· Monitoring: https://your-domain.com:3000

Default credentials:

· Owner: username: owner, password: ChangeMe123!
· Admin: username: admin, password: Admin123!

IMPORTANT: Change these passwords immediately after first login!