# backend/admin_panel/controllers/analytics_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from datetime import datetime, date, timedelta
import logging

from core.utils import datetime_utils
from shared.constants import KeyDuration

logger = logging.getLogger(__name__)

class AnalyticsController:
    """Controller for analytics and reporting"""
    
    async def get_overview(self, reserve_id: str, period: str,
                          db: Session) -> Dict[str, Any]:
        """Get analytics overview"""
        now = datetime_utils.now()
        
        if period == 'day':
            start = now.replace(hour=0, minute=0, second=0, microsecond=0)
        elif period == 'week':
            start = now - timedelta(days=7)
        elif period == 'month':
            start = now - timedelta(days=30)
        elif period == 'year':
            start = now - timedelta(days=365)
        else:
            start = now - timedelta(days=7)
        
        return {
            'period': period,
            'start_date': start.isoformat(),
            'end_date': now.isoformat(),
            'users': {
                'total': 5678,
                'new': 234,
                'active_daily': 1234,
                'active_weekly': 3456,
                'active_monthly': 4567,
                'retention_rate': 68.5
            },
            'keys': {
                'sold': 1234,
                'revenue': 740400,
                'avg_price': 600,
                'popular_duration': '30d',
                'by_duration': {
                    '1d': 234,
                    '3d': 156,
                    '7d': 345,
                    '30d': 456,
                    '60d': 43
                }
            },
            'payments': {
                'successful': 1185,
                'failed': 49,
                'success_rate': 96.0,
                'total_amount': 740400,
                'avg_amount': 600
            },
            'activity': {
                'total_actions': 23456,
                'unique_users': 3456,
                'top_actions': [
                    {'action': 'start', 'count': 12345},
                    {'action': 'view_plans', 'count': 5678},
                    {'action': 'purchase', 'count': 1234},
                    {'action': 'download', 'count': 987},
                    {'action': 'view_keys', 'count': 876}
                ],
                'peak_hour': 20
            },
            'growth': [
                {
                    'date': (now - timedelta(days=i)).strftime('%Y-%m-%d'),
                    'users': 5000 + i * 10,
                    'revenue': 600000 + i * 5000,
                    'keys': 1000 + i * 5
                }
                for i in range(7)
            ]
        }
    
    async def get_user_analytics(self, reserve_id: str, date_from: date,
                                date_to: date, db: Session) -> Dict[str, Any]:
        """Get detailed user analytics"""
        return {
            'period': {
                'from': date_from.isoformat(),
                'to': date_to.isoformat()
            },
            'acquisition': {
                'new_users': 1234,
                'by_source': {
                    'direct': 800,
                    'referral': 300,
                    'social': 134
                },
                'daily_new': [
                    {
                        'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                        'count': 50 - i
                    }
                    for i in range(7)
                ]
            },
            'engagement': {
                'daily_active': 456,
                'weekly_active': 2345,
                'monthly_active': 4567,
                'sessions_per_user': 3.2,
                'avg_session_duration': 180,  # seconds
                'retention': {
                    'day_1': 65.5,
                    'day_7': 45.2,
                    'day_30': 28.7
                }
            },
            'segments': {
                'premium_users': 567,
                'inactive_users': 1234,
                'new_users': 234,
                'returning_users': 876
            },
            'top_users': [
                {
                    'user_id': 123456,
                    'username': 'power_user',
                    'purchases': 25,
                    'spent': 15000,
                    'last_active': datetime_utils.now().isoformat()
                }
                for _ in range(5)
            ]
        }
    
    async def get_revenue_analytics(self, reserve_id: str, date_from: date,
                                   date_to: date, group_by: str,
                                   db: Session) -> Dict[str, Any]:
        """Get revenue analytics"""
        return {
            'period': {
                'from': date_from.isoformat(),
                'to': date_to.isoformat()
            },
            'summary': {
                'total_revenue': 750000,
                'total_transactions': 1250,
                'avg_transaction': 600,
                'max_transaction': 800,
                'min_transaction': 100
            },
            'by_duration': {
                '1d': {'count': 250, 'revenue': 25000},
                '3d': {'count': 200, 'revenue': 50000},
                '7d': {'count': 300, 'revenue': 105000},
                '30d': {'count': 400, 'revenue': 240000},
                '60d': {'count': 100, 'revenue': 80000}
            },
            'by_payment_method': {
                'razorpay': {'count': 800, 'revenue': 480000},
                'phonepe': {'count': 300, 'revenue': 180000},
                'paytm': {'count': 150, 'revenue': 90000}
            },
            'trends': [
                {
                    'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                    'revenue': 50000 - i * 2000,
                    'transactions': 80 - i * 3
                }
                for i in range(7)
            ]
        }
    
    async def get_key_analytics(self, reserve_id: str, date_from: date,
                               date_to: date, db: Session) -> Dict[str, Any]:
        """Get key sales analytics"""
        return {
            'period': {
                'from': date_from.isoformat(),
                'to': date_to.isoformat()
            },
            'summary': {
                'total_sold': 1250,
                'total_revenue': 750000,
                'avg_price': 600,
                'most_popular': '30d'
            },
            'by_duration': {
                '1d': {'sold': 250, 'revenue': 25000, 'avg_price': 100},
                '3d': {'sold': 200, 'revenue': 50000, 'avg_price': 250},
                '7d': {'sold': 300, 'revenue': 105000, 'avg_price': 350},
                '30d': {'sold': 400, 'revenue': 240000, 'avg_price': 600},
                '60d': {'sold': 100, 'revenue': 80000, 'avg_price': 800}
            },
            'inventory': {
                'total_available': 5000,
                'by_duration': {
                    '1d': 1200,
                    '3d': 800,
                    '7d': 1500,
                    '30d': 1000,
                    '60d': 500
                },
                'low_stock': ['60d']
            },
            'sales_trend': [
                {
                    'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                    'keys_sold': 50 - i * 2,
                    'revenue': 30000 - i * 1200
                }
                for i in range(7)
            ]
        }
    
    async def get_dashboard(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get admin dashboard data"""
        now = datetime_utils.now()
        today_start = now.replace(hour=0, minute=0, second=0, microsecond=0)
        
        return {
            'stats': {
                'total_users': 5678,
                'new_today': 23,
                'active_today': 456,
                'keys_sold_today': 45,
                'revenue_today': 27000,
                'total_revenue': 750000,
                'expiry_days': 25
            },
            'recent_activity': [
                {
                    'user_id': 123456,
                    'username': 'john_doe',
                    'action': 'purchase',
                    'details': {'amount': 600},
                    'time': (now - timedelta(minutes=5)).isoformat()
                },
                {
                    'user_id': 123457,
                    'username': 'jane_doe',
                    'action': 'download',
                    'details': {'version': '1.1.0'},
                    'time': (now - timedelta(minutes=12)).isoformat()
                },
                {
                    'user_id': 123458,
                    'username': 'bob_smith',
                    'action': 'start',
                    'details': {},
                    'time': (now - timedelta(minutes=18)).isoformat()
                }
            ],
            'key_inventory': {
                '1d': 1200,
                '3d': 800,
                '7d': 1500,
                '30d': 1000,
                '60d': 500
            },
            'chart_data': {
                'sales': {
                    'labels': ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    'values': [45, 52, 48, 55, 62, 78, 65]
                },
                'revenue': {
                    'labels': ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    'values': [27000, 31200, 28800, 33000, 37200, 46800, 39000]
                },
                'users': {
                    'labels': ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    'values': [23, 25, 22, 28, 30, 35, 32]
                }
            }
        }