# backend/admin_panel/controllers/apk_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from fastapi import UploadFile
import logging
import hashlib
import os
from datetime import datetime

from core.exceptions import NotFoundError, ValidationError
from core.utils import datetime_utils, file_utils, id_generator
from shared.constants import MAX_FILE_SIZES, ALLOWED_EXTENSIONS

logger = logging.getLogger(__name__)

class APKController:
    """Controller for APK management"""
    
    async def upload_apk(self, reserve_id: str, file: UploadFile, 
                         version_name: str, version_code: int,
                         release_notes: Optional[str], is_mandatory: bool,
                         db: Session) -> Dict[str, Any]:
        """Upload new APK version"""
        # Validate file
        if not file_utils.validate_file_extension(file.filename, ALLOWED_EXTENSIONS['apk']):
            raise ValidationError(f"Invalid file type. Allowed: {ALLOWED_EXTENSIONS['apk']}")
        
        # Check file size
        file_size = 0
        content = await file.read()
        file_size = len(content)
        
        if not file_utils.validate_file_size(file_size, MAX_FILE_SIZES['apk']):
            raise ValidationError(f"File too large. Max size: {MAX_FILE_SIZES['apk'] / 1024 / 1024}MB")
        
        # Calculate file hash
        file_hash = hashlib.sha256(content).hexdigest()
        
        # Generate filename
        safe_filename = file_utils.generate_filename(f"apk_{version_code}", 'apk')
        
        # Save file (in production, this would save to cloud storage)
        save_path = f"/app/uploads/apk/{reserve_id}/{safe_filename}"
        os.makedirs(os.path.dirname(save_path), exist_ok=True)
        
        with open(save_path, 'wb') as f:
            f.write(content)
        
        # Create version record (this would save to tenant database)
        version = {
            'id': id_generator.generate_batch_id(),
            'version_name': version_name,
            'version_code': version_code,
            'file_name': safe_filename,
            'file_path': save_path,
            'file_size': file_size,
            'file_hash': file_hash,
            'release_notes': release_notes,
            'is_mandatory': is_mandatory,
            'is_active': False,
            'uploaded_at': datetime_utils.now().isoformat()
        }
        
        logger.info(f"APK uploaded for {reserve_id}: v{version_name} ({version_code})")
        
        return version
    
    async def get_versions(self, reserve_id: str, db: Session) -> List[Dict[str, Any]]:
        """Get all APK versions"""
        # This would query the tenant database
        return [
            {
                'id': 1,
                'version_name': '1.0.0',
                'version_code': 1,
                'file_size': 45 * 1024 * 1024,
                'release_notes': 'Initial release',
                'download_count': 150,
                'unique_downloads': 120,
                'is_active': True,
                'is_mandatory': False,
                'uploaded_at': datetime_utils.now().isoformat()
            },
            {
                'id': 2,
                'version_name': '1.1.0',
                'version_code': 2,
                'file_size': 46 * 1024 * 1024,
                'release_notes': 'Bug fixes and improvements',
                'download_count': 75,
                'unique_downloads': 60,
                'is_active': False,
                'is_mandatory': False,
                'uploaded_at': datetime_utils.now().isoformat()
            }
        ]
    
    async def activate_version(self, reserve_id: str, version_code: int,
                              db: Session) -> Dict[str, Any]:
        """Set active APK version"""
        # This would update the tenant database
        return {
            'reserve_id': reserve_id,
            'version_code': version_code,
            'is_active': True,
            'activated_at': datetime_utils.now().isoformat()
        }
    
    async def get_stats(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get APK download statistics"""
        return {
            'total_downloads': 225,
            'unique_downloads': 180,
            'by_version': [
                {
                    'version': '1.0.0',
                    'downloads': 150,
                    'unique': 120
                },
                {
                    'version': '1.1.0',
                    'downloads': 75,
                    'unique': 60
                }
            ],
            'daily_downloads': [
                {
                    'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                    'count': 15 - i
                }
                for i in range(7)
            ]
        }
    
    async def delete_version(self, reserve_id: str, version_code: int,
                            db: Session) -> Dict[str, Any]:
        """Delete APK version"""
        # This would delete from tenant database and storage
        logger.info(f"APK version {version_code} deleted for {reserve_id}")
        
        return {
            'reserve_id': reserve_id,
            'version_code': version_code,
            'deleted': True,
            'deleted_at': datetime_utils.now().isoformat()
        }