# backend/admin_panel/controllers/broadcast_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from datetime import datetime
import logging

from core.exceptions import NotFoundError, ValidationError, BroadcastError
from core.utils import datetime_utils, id_generator
from admin_panel.services.broadcast_service import BroadcastService
from shared.constants import BroadcastStatus, BroadcastTarget
from tasks.broadcast_tasks import process_broadcast

logger = logging.getLogger(__name__)

class BroadcastController:
    """Controller for broadcast management"""
    
    def __init__(self):
        self.broadcast_service = BroadcastService()
    
    async def create_broadcast(self, reserve_id: str, admin_id: int,
                              broadcast_data: Dict[str, Any],
                              db: Session) -> Dict[str, Any]:
        """Create and send broadcast"""
        broadcast_id = id_generator.generate_broadcast_id()
        
        # Prepare broadcast record
        broadcast = {
            'broadcast_id': broadcast_id,
            'reserve_id': reserve_id,
            'admin_id': admin_id,
            'message': broadcast_data['message'],
            'preview_text': broadcast_data.get('preview_text'),
            'media_url': broadcast_data.get('media_url'),
            'buttons': broadcast_data.get('buttons', []),
            'target_type': broadcast_data['target_type'],
            'target_filters': broadcast_data.get('target_filters', {}),
            'scheduled_for': broadcast_data.get('scheduled_for'),
            'status': 'draft' if broadcast_data.get('scheduled_for') else 'sending',
            'created_at': datetime_utils.now().isoformat()
        }
        
        # Get target users based on filters
        target_user_ids = await self._get_target_users(
            reserve_id, 
            broadcast_data['target_type'],
            broadcast_data.get('target_filters', {}),
            db
        )
        
        broadcast['total_recipients'] = len(target_user_ids)
        
        # Save to database (this would save to tenant database)
        
        # If not scheduled, send immediately
        if not broadcast_data.get('scheduled_for'):
            # Trigger async broadcast task
            process_broadcast.delay(
                broadcast_id=broadcast_id,
                admin_id=admin_id,
                reserve_id=reserve_id,
                message=broadcast_data['message'],
                target_user_ids=target_user_ids,
                parse_mode='HTML'
            )
            
            broadcast['status'] = 'sending'
            logger.info(f"Broadcast {broadcast_id} started for {reserve_id} to {len(target_user_ids)} users")
        else:
            logger.info(f"Broadcast {broadcast_id} scheduled for {broadcast_data['scheduled_for']}")
        
        return broadcast
    
    async def get_history(self, reserve_id: str, skip: int, limit: int,
                         db: Session) -> Dict[str, Any]:
        """Get broadcast history"""
        # This would query the tenant database
        return {
            'total': 25,
            'broadcasts': [
                {
                    'broadcast_id': f'BCST{i:04d}',
                    'message': f'Test broadcast {i}',
                    'target_type': 'all',
                    'total_recipients': 1000,
                    'delivered_count': 980,
                    'failed_count': 20,
                    'opened_count': 450,
                    'status': 'completed',
                    'created_at': datetime_utils.now().isoformat(),
                    'sent_at': datetime_utils.now().isoformat()
                }
                for i in range(skip, min(skip + limit, 25))
            ]
        }
    
    async def get_stats(self, reserve_id: str, broadcast_id: str,
                       db: Session) -> Dict[str, Any]:
        """Get broadcast delivery statistics"""
        return {
            'broadcast_id': broadcast_id,
            'total_recipients': 1000,
            'delivered_count': 980,
            'failed_count': 20,
            'opened_count': 450,
            'opened_rate': 45.9,
            'delivery_rate': 98.0,
            'failed_users': [
                {'user_id': 123456, 'reason': 'User blocked bot'},
                {'user_id': 123457, 'reason': 'Chat not found'}
            ],
            'hourly_opens': [
                {'hour': 9, 'opens': 50},
                {'hour': 10, 'opens': 120},
                {'hour': 11, 'opens': 150},
                {'hour': 12, 'opens': 130}
            ]
        }
    
    async def cancel_broadcast(self, reserve_id: str, broadcast_id: str,
                              db: Session) -> Dict[str, Any]:
        """Cancel scheduled broadcast"""
        # This would update the tenant database
        return {
            'broadcast_id': broadcast_id,
            'status': 'cancelled',
            'cancelled_at': datetime_utils.now().isoformat()
        }
    
    async def _get_target_users(self, reserve_id: str, target_type: str,
                               filters: Dict[str, Any], db: Session) -> List[int]:
        """Get list of target user IDs based on filters"""
        # This would query the tenant database
        # For now, return mock data
        if target_type == BroadcastTarget.ALL.value:
            return [1000000 + i for i in range(1000)]
        elif target_type == BroadcastTarget.PREMIUM.value:
            return [1000000 + i for i in range(200)]
        elif target_type == BroadcastTarget.ACTIVE.value:
            return [1000000 + i for i in range(500)]
        elif target_type == BroadcastTarget.INACTIVE.value:
            return [1000000 + i for i in range(300)]
        else:
            return []