# backend/admin_panel/controllers/key_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from fastapi import UploadFile
import logging
import csv
import io

from core.exceptions import NotFoundError, ValidationError, KeyGenerationError
from core.utils import key_generator, datetime_utils
from admin_panel.services.key_service import KeyService
from shared.constants import KeyType, KeyStatus, KeyDuration

logger = logging.getLogger(__name__)

class KeyController:
    """Controller for key management"""
    
    def __init__(self):
        self.key_service = KeyService()
    
    async def upload_stored_keys(self, reserve_id: str, file: UploadFile, 
                                 db: Session) -> Dict[str, Any]:
        """Upload CSV file with stored keys"""
        # Read file content
        content = await file.read()
        
        # Process keys
        result = await self.key_service.upload_stored_keys(
            reserve_id, content, file.filename, db
        )
        
        logger.info(f"Stored keys uploaded for {reserve_id}: {result['valid_keys']} valid keys")
        
        return result
    
    async def get_stored_keys(self, reserve_id: str, duration: Optional[str],
                             status: Optional[str], batch_id: Optional[str],
                             skip: int, limit: int, db: Session) -> Dict[str, Any]:
        """Get stored keys inventory"""
        keys = await self.key_service.get_keys(
            reserve_id, KeyType.STORED.value, duration, status, None, skip, limit, db
        )
        
        # Get summary by duration
        summary = {}
        for dur in KeyDuration:
            count = await self.key_service.count_available_keys(reserve_id, dur.value, db)
            summary[dur.value] = count
        
        return {
            'total': keys.get('total', 0),
            'keys': keys.get('keys', []),
            'summary': summary
        }
    
    async def delete_batch(self, reserve_id: str, batch_id: str, 
                          db: Session) -> Dict[str, Any]:
        """Delete entire key batch"""
        result = await self.key_service.delete_batch(reserve_id, batch_id, db)
        logger.info(f"Batch {batch_id} deleted for {reserve_id}")
        return result
    
    async def export_stored_keys(self, reserve_id: str, format: str,
                                duration: Optional[str], status: Optional[str],
                                db: Session) -> io.StringIO:
        """Export stored keys to CSV"""
        return await self.key_service.export_keys(
            reserve_id, format, duration, status, db
        )
    
    async def set_key_pattern(self, reserve_id: str, pattern_data: Dict[str, Any],
                             db: Session) -> Dict[str, Any]:
        """Set key generation pattern"""
        result = await self.key_service.set_key_pattern(reserve_id, pattern_data, db)
        logger.info(f"Key pattern updated for {reserve_id}")
        return result
    
    async def get_key_pattern(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get current key generation pattern"""
        return await self.key_service.get_key_pattern(reserve_id, db)
    
    async def test_pattern(self, reserve_id: str, count: int, 
                          db: Session) -> Dict[str, Any]:
        """Test key generation pattern"""
        return await self.key_service.test_pattern(reserve_id, count, db)
    
    async def toggle_auto_generation(self, reserve_id: str, enable: bool,
                                     db: Session) -> Dict[str, Any]:
        """Enable or disable auto key generation"""
        # This would update the tenant configuration
        return {
            'reserve_id': reserve_id,
            'auto_generation_enabled': enable,
            'updated_at': datetime_utils.now().isoformat()
        }
    
    async def get_all_keys(self, reserve_id: str, key_type: Optional[str],
                          duration: Optional[str], status: Optional[str],
                          user_id: Optional[int], skip: int, limit: int,
                          db: Session) -> Dict[str, Any]:
        """Get all keys with filters"""
        return await self.key_service.get_keys(
            reserve_id, key_type, duration, status, user_id, skip, limit, db
        )
    
    async def block_key(self, reserve_id: str, key_id: int, reason: str,
                       db: Session) -> Dict[str, Any]:
        """Block specific key"""
        result = await self.key_service.block_key(reserve_id, key_id, reason, db)
        logger.info(f"Key {key_id} blocked in {reserve_id}: {reason}")
        return result
    
    async def unblock_key(self, reserve_id: str, key_id: int, 
                         db: Session) -> Dict[str, Any]:
        """Unblock key"""
        result = await self.key_service.unblock_key(reserve_id, key_id, db)
        logger.info(f"Key {key_id} unblocked in {reserve_id}")
        return result
    
    async def extend_key(self, reserve_id: str, key_id: int, days: int,
                        db: Session) -> Dict[str, Any]:
        """Extend key validity"""
        result = await self.key_service.extend_key(reserve_id, key_id, days, db)
        logger.info(f"Key {key_id} extended by {days} days in {reserve_id}")
        return result
    
    async def get_key_stats(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get key statistics"""
        return await self.key_service.get_key_stats(reserve_id, db)