# backend/admin_panel/controllers/payment_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from datetime import datetime, date, timedelta
import logging

from core.exceptions import NotFoundError, ValidationError, PaymentError
from core.utils import datetime_utils, id_generator
from admin_panel.services.payment_service import PaymentService
from shared.constants import TransactionStatus, PaymentGateway

logger = logging.getLogger(__name__)

class PaymentController:
    """Controller for payment management"""
    
    def __init__(self):
        self.payment_service = PaymentService()
    
    async def get_transactions(self, reserve_id: str, status: Optional[str],
                              user_id: Optional[int], date_from: Optional[date],
                              date_to: Optional[date], skip: int, limit: int,
                              db: Session) -> Dict[str, Any]:
        """Get all transactions"""
        # This would query the tenant database
        return {
            'total': 100,
            'transactions': [
                {
                    'id': i,
                    'transaction_id': id_generator.generate_transaction_id(),
                    'user_id': user_id or 123456,
                    'amount': 600,
                    'currency': 'INR',
                    'status': 'success',
                    'payment_method': 'razorpay',
                    'created_at': datetime_utils.now().isoformat()
                }
                for i in range(skip, min(skip + limit, 100))
            ]
        }
    
    async def get_transaction(self, reserve_id: str, transaction_id: str,
                             db: Session) -> Dict[str, Any]:
        """Get transaction details"""
        return {
            'transaction_id': transaction_id,
            'user_id': 123456,
            'username': 'john_doe',
            'amount': 600,
            'currency': 'INR',
            'status': 'success',
            'payment_method': 'razorpay',
            'gateway_transaction_id': 'pay_123456',
            'key_delivered': 'GAME-ABCD-1234',
            'created_at': datetime_utils.now().isoformat(),
            'completed_at': datetime_utils.now().isoformat()
        }
    
    async def process_refund(self, reserve_id: str, transaction_id: str,
                            amount: Optional[float], reason: str,
                            db: Session) -> Dict[str, Any]:
        """Process refund for transaction"""
        # Get transaction
        transaction = await self.get_transaction(reserve_id, transaction_id, db)
        
        if not transaction:
            raise NotFoundError(f"Transaction {transaction_id} not found")
        
        if transaction['status'] != 'success':
            raise ValidationError("Can only refund successful transactions")
        
        refund_amount = amount or transaction['amount']
        
        # Process refund through payment gateway
        refund_result = await self.payment_service.process_refund(
            reserve_id,
            transaction['gateway_transaction_id'],
            refund_amount,
            reason
        )
        
        logger.info(f"Refund processed for transaction {transaction_id}: {refund_amount}")
        
        return {
            'transaction_id': transaction_id,
            'refund_id': refund_result.get('refund_id'),
            'amount': refund_amount,
            'reason': reason,
            'status': 'completed',
            'processed_at': datetime_utils.now().isoformat()
        }
    
    async def get_payment_stats(self, reserve_id: str, date_from: date,
                               date_to: date, db: Session) -> Dict[str, Any]:
        """Get payment statistics"""
        return {
            'period': {
                'from': date_from.isoformat(),
                'to': date_to.isoformat()
            },
            'total_transactions': 150,
            'successful': 145,
            'failed': 5,
            'refunded': 2,
            'total_amount': 87000,
            'successful_amount': 85000,
            'refunded_amount': 2000,
            'average_amount': 586,
            'success_rate': 96.67,
            'by_payment_method': {
                'razorpay': {'count': 100, 'amount': 60000},
                'phonepe': {'count': 30, 'amount': 18000},
                'paytm': {'count': 15, 'amount': 7000},
                'stripe': {'count': 5, 'amount': 2000}
            },
            'daily_breakdown': [
                {
                    'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                    'count': 10 - i,
                    'amount': 6000 - (i * 600)
                }
                for i in range(7)
            ]
        }
    
    async def configure_gateway(self, admin_id: int, gateway_config: Dict[str, Any],
                               db: Session) -> Dict[str, Any]:
        """Configure payment gateway for admin"""
        # This would update the admin record
        from core.models import Admin
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if not admin:
            raise NotFoundError(f"Admin {admin_id} not found")
        
        # Update payment gateway configuration
        admin.payment_gateway = gateway_config.get('gateway', admin.payment_gateway)
        admin.payment_api_key = gateway_config.get('api_key', admin.payment_api_key)
        admin.payment_api_secret = gateway_config.get('api_secret', admin.payment_api_secret)
        admin.payment_merchant_id = gateway_config.get('merchant_id', admin.payment_merchant_id)
        admin.payment_currency = gateway_config.get('currency', admin.payment_currency)
        
        db.commit()
        
        logger.info(f"Payment gateway configured for admin {admin_id}")
        
        return {
            'admin_id': admin_id,
            'gateway': admin.payment_gateway,
            'currency': admin.payment_currency,
            'configured_at': datetime_utils.now().isoformat()
        }