# backend/admin_panel/controllers/user_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from datetime import datetime, date, timedelta
import logging
import csv
import io
from fastapi.responses import StreamingResponse

from core.exceptions import NotFoundError, ValidationError
from core.utils import datetime_utils
from shared.constants import ActivityType

logger = logging.getLogger(__name__)

class UserController:
    """Controller for user management at admin level"""
    
    async def get_users(self, reserve_id: str, skip: int, limit: int, 
                       search: Optional[str], status: Optional[str],
                       db: Session) -> Dict[str, Any]:
        """Get users with filters"""
        # This would query the tenant database
        # For now, return mock data
        return {
            'total': 100,
            'users': [
                {
                    'id': i,
                    'telegram_id': 1000000 + i,
                    'username': f'user_{i}',
                    'first_name': f'First{i}',
                    'last_name': f'Last{i}',
                    'joined_at': datetime_utils.now().isoformat(),
                    'last_active': datetime_utils.now().isoformat(),
                    'total_purchases': i % 5,
                    'total_spent': (i % 5) * 100,
                    'status': 'active'
                }
                for i in range(skip, min(skip + limit, 100))
            ]
        }
    
    async def get_user_details(self, reserve_id: str, telegram_id: int, 
                              db: Session) -> Dict[str, Any]:
        """Get detailed user information"""
        # This would query the tenant database
        return {
            'telegram_id': telegram_id,
            'username': f'user_{telegram_id}',
            'first_name': 'John',
            'last_name': 'Doe',
            'joined_at': datetime_utils.now().isoformat(),
            'last_active': datetime_utils.now().isoformat(),
            'total_purchases': 5,
            'total_spent': 2500,
            'active_keys_count': 2,
            'status': 'active',
            'keys': [
                {
                    'key': 'GAME-ABCD-1234',
                    'duration': '30d',
                    'purchased_at': datetime_utils.now().isoformat(),
                    'expires_at': (datetime_utils.now() + timedelta(days=30)).isoformat()
                }
            ],
            'recent_activity': [
                {
                    'type': 'purchase',
                    'details': {'amount': 600},
                    'created_at': datetime_utils.now().isoformat()
                }
            ]
        }
    
    async def block_user(self, reserve_id: str, telegram_id: int, 
                        reason: str, db: Session) -> Dict[str, Any]:
        """Block user from bot"""
        # This would update the tenant database
        return {
            'telegram_id': telegram_id,
            'status': 'blocked',
            'reason': reason,
            'blocked_at': datetime_utils.now().isoformat()
        }
    
    async def unblock_user(self, reserve_id: str, telegram_id: int, 
                          db: Session) -> Dict[str, Any]:
        """Unblock user"""
        # This would update the tenant database
        return {
            'telegram_id': telegram_id,
            'status': 'active',
            'unblocked_at': datetime_utils.now().isoformat()
        }
    
    async def get_activity(self, reserve_id: str, user_id: Optional[int],
                          activity_type: Optional[str], date_from: Optional[date],
                          date_to: Optional[date], skip: int, limit: int,
                          db: Session) -> Dict[str, Any]:
        """Get user activity logs"""
        # This would query the tenant database
        return {
            'total': 50,
            'activities': [
                {
                    'id': i,
                    'user_id': user_id or 123456,
                    'activity_type': activity_type or 'purchase',
                    'details': {'amount': 600},
                    'ip_address': '192.168.1.1',
                    'created_at': datetime_utils.now().isoformat()
                }
                for i in range(skip, min(skip + limit, 50))
            ]
        }
    
    async def get_activity_stats(self, reserve_id: str, date_from: date,
                                date_to: date, db: Session) -> Dict[str, Any]:
        """Get activity statistics"""
        return {
            'total': 150,
            'by_type': {
                'purchase': 50,
                'download': 30,
                'start': 70
            },
            'by_hour': {
                '0': 5, '1': 3, '2': 2, '3': 1, '4': 1, '5': 2,
                '6': 4, '7': 8, '8': 12, '9': 15, '10': 18, '11': 20,
                '12': 22, '13': 25, '14': 28, '15': 30, '16': 32, '17': 35,
                '18': 38, '19': 40, '20': 42, '21': 45, '22': 48, '23': 50
            },
            'unique_users': 45
        }
    
    async def export_users(self, reserve_id: str, format: str, 
                          db: Session) -> StreamingResponse:
        """Export users to CSV"""
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow([
            'Telegram ID', 'Username', 'First Name', 'Last Name',
            'Joined Date', 'Last Active', 'Total Purchases',
            'Total Spent', 'Status'
        ])
        
        # Get users (this would query the tenant database)
        users = await self.get_users(reserve_id, 0, 1000, None, None, db)
        
        for user in users['users']:
            writer.writerow([
                user['telegram_id'],
                user.get('username', ''),
                user.get('first_name', ''),
                user.get('last_name', ''),
                user.get('joined_at', ''),
                user.get('last_active', ''),
                user.get('total_purchases', 0),
                user.get('total_spent', 0),
                user.get('status', 'active')
            ])
        
        output.seek(0)
        
        filename = f"users_{reserve_id}_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
        
        return StreamingResponse(
            iter([output.getvalue()]),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={filename}"}
        )
    
    async def export_activity(self, reserve_id: str, format: str,
                             date_from: Optional[date], date_to: Optional[date],
                             db: Session) -> StreamingResponse:
        """Export activity logs to CSV"""
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow([
            'User ID', 'Username', 'Activity Type', 'Details',
            'IP Address', 'Created At'
        ])
        
        # Get activities (this would query the tenant database)
        activities = await self.get_activity(
            reserve_id, None, None, date_from, date_to, 0, 1000, db
        )
        
        for act in activities['activities']:
            writer.writerow([
                act['user_id'],
                act.get('username', ''),
                act['activity_type'],
                str(act['details']),
                act.get('ip_address', ''),
                act['created_at']
            ])
        
        output.seek(0)
        
        filename = f"activity_{reserve_id}_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
        
        return StreamingResponse(
            iter([output.getvalue()]),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={filename}"}
        )