# backend/admin_panel/routes.py
from fastapi import APIRouter, Depends, HTTPException, Query, UploadFile, File
from typing import List, Optional, Dict, Any
from datetime import date, datetime

from core.database import get_db
from core.security import get_current_admin
from shared.decorators.role_required import admin_required
from shared.decorators.tenant_required import tenant_required
from .controllers import (
    UserController,
    KeyController,
    PaymentController,
    APKController,
    BroadcastController,
    AnalyticsController
)
from .schemas import (
    UserResponse, UserListResponse,
    KeyCreate, KeyResponse, KeyListResponse,
    KeyPatternCreate, KeyPatternResponse,
    TransactionResponse, TransactionListResponse,
    APKVersionCreate, APKVersionResponse,
    BroadcastCreate, BroadcastResponse,
    AnalyticsResponse
)

router = APIRouter(prefix="/admin", tags=["Admin Panel"])

# ==================== User Management ====================

@router.get("/users", response_model=UserListResponse)
@admin_required
@tenant_required
async def get_my_users(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    search: Optional[str] = None,
    status: Optional[str] = None,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get all users of current admin's bot"""
    return await user_controller.get_users(
        current_admin.reserve_id, skip, limit, search, status, db
    )

@router.get("/users/{telegram_id}", response_model=UserResponse)
@admin_required
@tenant_required
async def get_user_details(
    telegram_id: int,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get detailed user information"""
    return await user_controller.get_user_details(
        current_admin.reserve_id, telegram_id, db
    )

@router.post("/users/{telegram_id}/block")
@admin_required
@tenant_required
async def block_user(
    telegram_id: int,
    reason: str = Query(...),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Block user from bot"""
    return await user_controller.block_user(
        current_admin.reserve_id, telegram_id, reason, db
    )

@router.post("/users/{telegram_id}/unblock")
@admin_required
@tenant_required
async def unblock_user(
    telegram_id: int,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Unblock user"""
    return await user_controller.unblock_user(
        current_admin.reserve_id, telegram_id, db
    )

@router.get("/users/export")
@admin_required
@tenant_required
async def export_users(
    format: str = Query("csv", regex="^(csv|pdf)$"),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Export users to CSV"""
    return await user_controller.export_users(current_admin.reserve_id, format, db)

# ==================== Activity Tracking ====================

@router.get("/activity")
@admin_required
@tenant_required
async def get_user_activity(
    user_id: Optional[int] = None,
    activity_type: Optional[str] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get user activity logs"""
    return await user_controller.get_activity(
        current_admin.reserve_id, user_id, activity_type, 
        date_from, date_to, skip, limit, db
    )

@router.get("/activity/stats")
@admin_required
@tenant_required
async def get_activity_stats(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get activity statistics"""
    return await user_controller.get_activity_stats(
        current_admin.reserve_id, date_from, date_to, db
    )

@router.get("/activity/export")
@admin_required
@tenant_required
async def export_activity(
    format: str = Query("csv", regex="^(csv|pdf)$"),
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Export activity logs"""
    return await user_controller.export_activity(
        current_admin.reserve_id, format, date_from, date_to, db
    )

# ==================== Stored Key Management ====================

@router.post("/keys/stored/upload")
@admin_required
@tenant_required
async def upload_stored_keys(
    file: UploadFile = File(...),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Upload CSV file with stored keys"""
    return await key_controller.upload_stored_keys(
        current_admin.reserve_id, file, db
    )

@router.get("/keys/stored", response_model=KeyListResponse)
@admin_required
@tenant_required
async def get_stored_keys(
    duration: Optional[str] = None,
    status: Optional[str] = None,
    batch_id: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Get stored keys inventory"""
    return await key_controller.get_stored_keys(
        current_admin.reserve_id, duration, status, batch_id, skip, limit, db
    )

@router.delete("/keys/stored/{batch_id}")
@admin_required
@tenant_required
async def delete_key_batch(
    batch_id: str,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Delete entire key batch"""
    return await key_controller.delete_batch(
        current_admin.reserve_id, batch_id, db
    )

@router.get("/keys/stored/export")
@admin_required
@tenant_required
async def export_stored_keys(
    format: str = Query("csv", regex="^(csv)$"),
    duration: Optional[str] = None,
    status: Optional[str] = None,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Export stored keys to CSV"""
    return await key_controller.export_stored_keys(
        current_admin.reserve_id, format, duration, status, db
    )

# ==================== Generated Key Management ====================

@router.post("/keys/generated/pattern")
@admin_required
@tenant_required
async def set_key_pattern(
    pattern_data: KeyPatternCreate,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Set key generation pattern"""
    return await key_controller.set_key_pattern(
        current_admin.reserve_id, pattern_data, db
    )

@router.get("/keys/generated/pattern", response_model=KeyPatternResponse)
@admin_required
@tenant_required
async def get_key_pattern(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Get current key generation pattern"""
    return await key_controller.get_key_pattern(current_admin.reserve_id, db)

@router.post("/keys/generated/test")
@admin_required
@tenant_required
async def test_key_pattern(
    count: int = Query(5, ge=1, le=20),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Test key generation pattern"""
    return await key_controller.test_pattern(current_admin.reserve_id, count, db)

@router.post("/keys/generated/enable")
@admin_required
@tenant_required
async def enable_auto_generation(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Enable auto key generation"""
    return await key_controller.toggle_auto_generation(
        current_admin.reserve_id, True, db
    )

@router.post("/keys/generated/disable")
@admin_required
@tenant_required
async def disable_auto_generation(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Disable auto key generation"""
    return await key_controller.toggle_auto_generation(
        current_admin.reserve_id, False, db
    )

# ==================== Common Key Operations ====================

@router.get("/keys", response_model=KeyListResponse)
@admin_required
@tenant_required
async def get_all_keys(
    key_type: Optional[str] = None,
    duration: Optional[str] = None,
    status: Optional[str] = None,
    user_id: Optional[int] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Get all keys (stored + generated)"""
    return await key_controller.get_all_keys(
        current_admin.reserve_id, key_type, duration, status, user_id, skip, limit, db
    )

@router.post("/keys/{key_id}/block")
@admin_required
@tenant_required
async def block_key(
    key_id: int,
    reason: str = Query(...),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Block specific key"""
    return await key_controller.block_key(
        current_admin.reserve_id, key_id, reason, db
    )

@router.post("/keys/{key_id}/unblock")
@admin_required
@tenant_required
async def unblock_key(
    key_id: int,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Unblock key"""
    return await key_controller.unblock_key(
        current_admin.reserve_id, key_id, db
    )

@router.post("/keys/{key_id}/extend")
@admin_required
@tenant_required
async def extend_key(
    key_id: int,
    days: int = Query(..., ge=1, le=365),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Extend key validity"""
    return await key_controller.extend_key(
        current_admin.reserve_id, key_id, days, db
    )

@router.get("/keys/stats")
@admin_required
@tenant_required
async def get_key_stats(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    key_controller: KeyController = Depends()
):
    """Get key statistics"""
    return await key_controller.get_key_stats(current_admin.reserve_id, db)

# ==================== Payment Management ====================

@router.get("/payments", response_model=TransactionListResponse)
@admin_required
@tenant_required
async def get_transactions(
    status: Optional[str] = None,
    user_id: Optional[int] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    payment_controller: PaymentController = Depends()
):
    """Get all transactions"""
    return await payment_controller.get_transactions(
        current_admin.reserve_id, status, user_id, date_from, date_to, skip, limit, db
    )

@router.get("/payments/{transaction_id}", response_model=TransactionResponse)
@admin_required
@tenant_required
async def get_transaction_details(
    transaction_id: str,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    payment_controller: PaymentController = Depends()
):
    """Get transaction details"""
    return await payment_controller.get_transaction(
        current_admin.reserve_id, transaction_id, db
    )

@router.post("/payments/{transaction_id}/refund")
@admin_required
@tenant_required
async def refund_transaction(
    transaction_id: str,
    amount: Optional[float] = None,
    reason: str = Query(...),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    payment_controller: PaymentController = Depends()
):
    """Process refund for transaction"""
    return await payment_controller.process_refund(
        current_admin.reserve_id, transaction_id, amount, reason, db
    )

@router.get("/payments/stats")
@admin_required
@tenant_required
async def get_payment_stats(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    payment_controller: PaymentController = Depends()
):
    """Get payment statistics"""
    return await payment_controller.get_payment_stats(
        current_admin.reserve_id, date_from, date_to, db
    )

@router.put("/payments/gateway")
@admin_required
@tenant_required
async def configure_payment_gateway(
    gateway_config: Dict[str, Any],
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    payment_controller: PaymentController = Depends()
):
    """Configure payment gateway"""
    return await payment_controller.configure_gateway(
        current_admin.id, gateway_config, db
    )

# ==================== APK Management ====================

@router.post("/apk/upload")
@admin_required
@tenant_required
async def upload_apk(
    file: UploadFile = File(...),
    version_name: str = Form(...),
    version_code: int = Form(...),
    release_notes: Optional[str] = Form(None),
    is_mandatory: bool = Form(False),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    apk_controller: APKController = Depends()
):
    """Upload new APK version"""
    return await apk_controller.upload_apk(
        current_admin.reserve_id, file, version_name, version_code,
        release_notes, is_mandatory, db
    )

@router.get("/apk/versions", response_model=List[APKVersionResponse])
@admin_required
@tenant_required
async def get_apk_versions(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    apk_controller: APKController = Depends()
):
    """Get all APK versions"""
    return await apk_controller.get_versions(current_admin.reserve_id, db)

@router.post("/apk/versions/{version_code}/activate")
@admin_required
@tenant_required
async def activate_apk_version(
    version_code: int,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    apk_controller: APKController = Depends()
):
    """Set active APK version"""
    return await apk_controller.activate_version(
        current_admin.reserve_id, version_code, db
    )

@router.get("/apk/stats")
@admin_required
@tenant_required
async def get_apk_stats(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    apk_controller: APKController = Depends()
):
    """Get APK download statistics"""
    return await apk_controller.get_stats(current_admin.reserve_id, db)

# ==================== Broadcast Management ====================

@router.post("/broadcast")
@admin_required
@tenant_required
async def create_broadcast(
    broadcast_data: BroadcastCreate,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    broadcast_controller: BroadcastController = Depends()
):
    """Create and send broadcast"""
    return await broadcast_controller.create_broadcast(
        current_admin.reserve_id, current_admin.id, broadcast_data, db
    )

@router.get("/broadcast/history", response_model=List[BroadcastResponse])
@admin_required
@tenant_required
async def get_broadcast_history(
    skip: int = Query(0, ge=0),
    limit: int = Query(50, ge=1, le=200),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    broadcast_controller: BroadcastController = Depends()
):
    """Get broadcast history"""
    return await broadcast_controller.get_history(
        current_admin.reserve_id, skip, limit, db
    )

@router.get("/broadcast/{broadcast_id}/stats")
@admin_required
@tenant_required
async def get_broadcast_stats(
    broadcast_id: str,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    broadcast_controller: BroadcastController = Depends()
):
    """Get broadcast delivery statistics"""
    return await broadcast_controller.get_stats(
        current_admin.reserve_id, broadcast_id, db
    )

@router.post("/broadcast/{broadcast_id}/cancel")
@admin_required
@tenant_required
async def cancel_scheduled_broadcast(
    broadcast_id: str,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    broadcast_controller: BroadcastController = Depends()
):
    """Cancel scheduled broadcast"""
    return await broadcast_controller.cancel_broadcast(
        current_admin.reserve_id, broadcast_id, db
    )

# ==================== Analytics ====================

@router.get("/analytics/overview", response_model=AnalyticsResponse)
@admin_required
@tenant_required
async def get_analytics_overview(
    period: str = Query("week", regex="^(day|week|month|year)$"),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    analytics_controller: AnalyticsController = Depends()
):
    """Get analytics overview"""
    return await analytics_controller.get_overview(
        current_admin.reserve_id, period, db
    )

@router.get("/analytics/users")
@admin_required
@tenant_required
async def get_user_analytics(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    analytics_controller: AnalyticsController = Depends()
):
    """Get detailed user analytics"""
    return await analytics_controller.get_user_analytics(
        current_admin.reserve_id, date_from, date_to, db
    )

@router.get("/analytics/revenue")
@admin_required
@tenant_required
async def get_revenue_analytics(
    date_from: date,
    date_to: date,
    group_by: str = Query("day", regex="^(day|week|month)$"),
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    analytics_controller: AnalyticsController = Depends()
):
    """Get revenue analytics"""
    return await analytics_controller.get_revenue_analytics(
        current_admin.reserve_id, date_from, date_to, group_by, db
    )

@router.get("/analytics/keys")
@admin_required
@tenant_required
async def get_key_analytics(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    analytics_controller: AnalyticsController = Depends()
):
    """Get key sales analytics"""
    return await analytics_controller.get_key_analytics(
        current_admin.reserve_id, date_from, date_to, db
    )

# ==================== Dashboard ====================

@router.get("/dashboard")
@admin_required
@tenant_required
async def get_dashboard(
    current_admin = Depends(get_current_admin),
    db = Depends(get_db),
    analytics_controller: AnalyticsController = Depends()
):
    """Get admin dashboard data"""
    return await analytics_controller.get_dashboard(current_admin.reserve_id, db)