# backend/admin_panel/schemas/admin_schema.py
from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Any
from datetime import datetime

class UserResponse(BaseModel):
    """Schema for user response"""
    id: int
    telegram_id: int
    username: Optional[str] = None
    first_name: Optional[str] = None
    last_name: Optional[str] = None
    joined_at: datetime
    last_active: Optional[datetime] = None
    total_purchases: int
    total_spent: float
    status: str
    
    class Config:
        from_attributes = True

class UserListResponse(BaseModel):
    """Schema for list of users response"""
    total: int
    users: List[UserResponse]

class TransactionResponse(BaseModel):
    """Schema for transaction response"""
    id: int
    transaction_id: str
    user_id: int
    amount: float
    currency: str
    status: str
    payment_method: str
    key_value: Optional[str] = None
    created_at: datetime
    completed_at: Optional[datetime] = None
    
    class Config:
        from_attributes = True

class TransactionListResponse(BaseModel):
    """Schema for list of transactions response"""
    total: int
    transactions: List[TransactionResponse]

class APKVersionResponse(BaseModel):
    """Schema for APK version response"""
    id: int
    version_name: str
    version_code: int
    file_size: int
    release_notes: Optional[str] = None
    download_count: int
    unique_downloads: int
    is_active: bool
    is_mandatory: bool
    uploaded_at: datetime
    
    class Config:
        from_attributes = True

class BroadcastResponse(BaseModel):
    """Schema for broadcast response"""
    broadcast_id: str
    message: str
    target_type: str
    total_recipients: int
    delivered_count: int
    failed_count: int
    opened_count: int
    status: str
    created_at: datetime
    sent_at: Optional[datetime] = None
    
    class Config:
        from_attributes = True

class BroadcastCreate(BaseModel):
    """Schema for creating broadcast"""
    message: str = Field(..., min_length=1, max_length=4096)
    target_type: str = Field(..., description="all, premium, active, inactive")
    target_filters: Optional[Dict[str, Any]] = Field(None)
    scheduled_for: Optional[datetime] = Field(None)
    media_url: Optional[str] = Field(None)
    buttons: Optional[List[Dict[str, str]]] = Field(None)
    
    @validator('target_type')
    def validate_target_type(cls, v):
        allowed = ['all', 'premium', 'active', 'inactive']
        if v not in allowed:
            raise ValueError(f"Target type must be one of {allowed}")
        return v