# backend/admin_panel/schemas/key_schema.py
from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Any
from datetime import datetime

class KeyCreate(BaseModel):
    """Schema for creating keys"""
    duration: str = Field(..., description="Key duration (1d, 3d, 7d, 30d, 60d)")
    count: int = Field(..., ge=1, le=1000, description="Number of keys to generate")
    pattern: Optional[Dict[str, Any]] = Field(None, description="Key generation pattern")
    price: Optional[float] = Field(None, description="Price for generated keys")
    
    @validator('duration')
    def validate_duration(cls, v):
        allowed = ['1d', '3d', '7d', '30d', '60d']
        if v not in allowed:
            raise ValueError(f"Duration must be one of {allowed}")
        return v

class KeyResponse(BaseModel):
    """Schema for key response"""
    id: int
    key_value: str
    key_type: str
    duration: str
    price: float
    status: str
    user_id: Optional[int] = None
    purchased_at: Optional[datetime] = None
    expires_at: Optional[datetime] = None
    created_at: datetime
    
    class Config:
        from_attributes = True

class KeyListResponse(BaseModel):
    """Schema for list of keys response"""
    total: int
    keys: List[KeyResponse]

class KeyPatternCreate(BaseModel):
    """Schema for creating key pattern"""
    pattern: str = Field(..., description="Key generation pattern")
    prefix: Optional[str] = Field(None, description="Key prefix")
    exclude_similar: bool = Field(True, description="Exclude similar characters")
    checksum_enabled: bool = Field(False, description="Enable checksum")
    duration: Optional[str] = Field(None, description="Associated duration")
    
    @validator('pattern')
    def validate_pattern(cls, v):
        if not v or len(v) < 3:
            raise ValueError("Pattern too short")
        return v

class KeyPatternResponse(BaseModel):
    """Schema for key pattern response"""
    id: int
    pattern: str
    prefix: Optional[str]
    exclude_similar: bool
    checksum_enabled: bool
    duration: Optional[str]
    is_default: bool
    is_active: bool
    created_at: datetime
    updated_at: datetime