# backend/admin_panel/services/broadcast_service.py
from typing import List, Dict, Any, Optional
import logging
from datetime import datetime

from core.utils import id_generator, datetime_utils
from tasks.broadcast_tasks import process_broadcast

logger = logging.getLogger(__name__)

class BroadcastService:
    """Service for broadcast management"""
    
    async def create_broadcast(self, reserve_id: str, admin_id: int,
                               message: str, target_type: str,
                               target_filters: Optional[Dict] = None,
                               scheduled_for: Optional[datetime] = None,
                               media_url: Optional[str] = None,
                               buttons: Optional[List[Dict]] = None) -> Dict[str, Any]:
        """Create new broadcast"""
        broadcast_id = id_generator.generate_broadcast_id()
        
        broadcast = {
            'broadcast_id': broadcast_id,
            'reserve_id': reserve_id,
            'admin_id': admin_id,
            'message': message,
            'media_url': media_url,
            'buttons': buttons or [],
            'target_type': target_type,
            'target_filters': target_filters or {},
            'scheduled_for': scheduled_for.isoformat() if scheduled_for else None,
            'status': 'scheduled' if scheduled_for else 'draft',
            'created_at': datetime_utils.now().isoformat()
        }
        
        logger.info(f"Broadcast {broadcast_id} created for {reserve_id}")
        return broadcast
    
    async def send_broadcast(self, broadcast_id: str, reserve_id: str,
                            admin_id: int, message: str,
                            target_user_ids: List[int]) -> Dict[str, Any]:
        """Send broadcast immediately"""
        # Trigger async task
        task = process_broadcast.delay(
            broadcast_id=broadcast_id,
            admin_id=admin_id,
            reserve_id=reserve_id,
            message=message,
            target_user_ids=target_user_ids
        )
        
        return {
            'broadcast_id': broadcast_id,
            'status': 'sending',
            'task_id': task.id,
            'total_recipients': len(target_user_ids)
        }
    
    async def get_target_users(self, reserve_id: str, target_type: str,
                              filters: Dict[str, Any]) -> List[int]:
        """Get list of target users based on filters"""
        # This would query tenant database
        # For now, return mock data
        if target_type == 'all':
            return [1000000 + i for i in range(1000)]
        elif target_type == 'premium':
            return [1000000 + i for i in range(200)]
        elif target_type == 'active':
            return [1000000 + i for i in range(500)]
        elif target_type == 'inactive':
            return [1000000 + i for i in range(300)]
        else:
            return []
    
    async def get_broadcast_stats(self, broadcast_id: str, reserve_id: str) -> Dict[str, Any]:
        """Get broadcast delivery statistics"""
        # This would query database
        return {
            'broadcast_id': broadcast_id,
            'total_recipients': 1000,
            'delivered': 980,
            'failed': 20,
            'opened': 450,
            'delivery_rate': 98.0,
            'open_rate': 45.9,
            'failed_users': [
                {'user_id': 123456, 'reason': 'Chat not found'},
                {'user_id': 123457, 'reason': 'User blocked bot'}
            ]
        }
    
    async def cancel_broadcast(self, broadcast_id: str, reserve_id: str) -> bool:
        """Cancel scheduled broadcast"""
        # This would update database
        logger.info(f"Broadcast {broadcast_id} cancelled for {reserve_id}")
        return True