# backend/admin_panel/services/key_service.py
from typing import List, Optional, Dict, Any, Tuple
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_, func
from datetime import datetime, timedelta
import logging
import csv
import io
import hashlib

from core.utils import key_generator, id_generator, datetime_utils
from core.exceptions import KeyGenerationError, ValidationError, NotFoundError
from shared.constants import KeyType, KeyStatus, KeyDuration
from config.settings import settings

logger = logging.getLogger(__name__)

class KeyService:
    """Service for key management operations"""
    
    def __init__(self):
        self.key_generator = key_generator
    
    async def upload_stored_keys(self, reserve_id: str, file_content: bytes, 
                                 filename: str, db: Session) -> Dict[str, Any]:
        """Process uploaded CSV file with stored keys"""
        try:
            # Parse CSV
            content = file_content.decode('utf-8')
            csv_reader = csv.reader(io.StringIO(content))
            
            # Skip header if present
            header = next(csv_reader)
            if header and not header[0].startswith('KEY'):
                # No header, rewind
                csv_reader = csv.reader(io.StringIO(content))
            
            # Process keys
            batch_id = id_generator.generate_batch_id()
            keys = []
            invalid_keys = []
            duplicates = []
            
            for row in csv_reader:
                if len(row) < 3:
                    invalid_keys.append({'row': row, 'reason': 'Invalid format'})
                    continue
                
                key_value, duration, price = row[0].strip(), row[1].strip(), float(row[2])
                
                # Validate key format
                if not self._validate_key_format(key_value):
                    invalid_keys.append({'key': key_value, 'reason': 'Invalid format'})
                    continue
                
                # Validate duration
                if duration not in [d.value for d in KeyDuration]:
                    invalid_keys.append({'key': key_value, 'reason': f'Invalid duration: {duration}'})
                    continue
                
                # Check for duplicates in this batch
                if key_value in [k['key'] for k in keys]:
                    duplicates.append(key_value)
                    continue
                
                # Check for duplicates in database
                # This would query the tenant database
                
                keys.append({
                    'key': key_value,
                    'duration': duration,
                    'price': price,
                    'type': KeyType.STORED.value
                })
            
            # Save to database (implementation would insert into tenant DB)
            # For now, just return summary
            
            return {
                'batch_id': batch_id,
                'filename': filename,
                'total_keys': len(keys) + len(invalid_keys) + len(duplicates),
                'valid_keys': len(keys),
                'invalid_keys': len(invalid_keys),
                'duplicates': len(duplicates),
                'invalid_keys_list': invalid_keys[:10] if invalid_keys else [],
                'keys_by_duration': self._count_by_duration(keys)
            }
            
        except Exception as e:
            logger.error(f"Error processing key upload: {e}")
            raise KeyGenerationError(f"Failed to process key file: {str(e)}")
    
    async def generate_keys(self, reserve_id: str, pattern: Dict[str, Any], 
                           duration: str, count: int, db: Session) -> Dict[str, Any]:
        """Generate keys based on pattern"""
        try:
            # Generate keys
            keys = self.key_generator.generate_batch(
                pattern=pattern.get('pattern', settings.KEY_DEFAULT_PATTERN),
                count=count,
                prefix=pattern.get('prefix', ''),
                exclude_similar=pattern.get('exclude_similar', True),
                enable_checksum=pattern.get('checksum_enabled', False)
            )
            
            # Prepare key records
            key_records = []
            for key in keys:
                key_records.append({
                    'key_value': key,
                    'key_type': KeyType.GENERATED.value,
                    'duration': duration,
                    'price': pattern.get('price', 0),
                    'status': KeyStatus.AVAILABLE.value,
                    'generation_rule_id': pattern.get('rule_id'),
                    'generation_date': datetime_utils.now()
                })
            
            # Save to database (implementation would insert into tenant DB)
            
            return {
                'generated': len(keys),
                'duration': duration,
                'sample_keys': keys[:5],
                'pattern_used': pattern.get('pattern')
            }
            
        except Exception as e:
            logger.error(f"Error generating keys: {e}")
            raise KeyGenerationError(f"Failed to generate keys: {str(e)}")
    
    async def get_key_details(self, reserve_id: str, key_id: int, db: Session) -> Dict[str, Any]:
        """Get detailed information about a specific key"""
        # Implementation would query tenant database
        return {}
    
    async def get_keys(self, reserve_id: str, key_type: Optional[str], duration: Optional[str],
                      status: Optional[str], user_id: Optional[int], skip: int, limit: int,
                      db: Session) -> Dict[str, Any]:
        """Get keys with filters"""
        # Implementation would query tenant database
        return {
            'total': 0,
            'keys': []
        }
    
    async def get_key_stats(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get key statistics"""
        # Implementation would query tenant database
        stats = {
            'total_keys': 0,
            'available_keys': 0,
            'sold_keys': 0,
            'blocked_keys': 0,
            'expired_keys': 0,
            'by_duration': {},
            'by_type': {},
            'low_stock_alerts': []
        }
        
        # Calculate low stock alerts
        for duration in KeyDuration:
            # Count available keys for this duration
            count = 0  # Would query database
            
            if count < settings.KEY_LOW_STOCK_ALERT:
                stats['low_stock_alerts'].append({
                    'duration': duration.value,
                    'count': count,
                    'threshold': settings.KEY_LOW_STOCK_ALERT
                })
        
        return stats
    
    async def block_key(self, reserve_id: str, key_id: int, reason: str, db: Session) -> Dict[str, Any]:
        """Block a specific key"""
        # Implementation would update tenant database
        return {
            'key_id': key_id,
            'status': 'blocked',
            'reason': reason
        }
    
    async def unblock_key(self, reserve_id: str, key_id: int, db: Session) -> Dict[str, Any]:
        """Unblock a specific key"""
        # Implementation would update tenant database
        return {
            'key_id': key_id,
            'status': 'available'
        }
    
    async def extend_key(self, reserve_id: str, key_id: int, days: int, db: Session) -> Dict[str, Any]:
        """Extend key validity"""
        # Implementation would update tenant database
        new_expiry = datetime_utils.get_expiry_date(days)
        
        return {
            'key_id': key_id,
            'days_added': days,
            'new_expiry': new_expiry.isoformat()
        }
    
    async def set_key_pattern(self, reserve_id: str, pattern_data: Dict[str, Any], 
                             db: Session) -> Dict[str, Any]:
        """Set key generation pattern"""
        # Validate pattern
        if not self._validate_pattern(pattern_data.get('pattern', '')):
            raise ValidationError("Invalid key pattern")
        
        # Save pattern (implementation would save to tenant DB)
        
        return {
            'message': 'Key pattern saved successfully',
            'pattern': pattern_data,
            'sample_keys': self.key_generator.generate_batch(
                pattern_data['pattern'],
                5,
                pattern_data.get('prefix', ''),
                pattern_data.get('exclude_similar', True),
                pattern_data.get('checksum_enabled', False)
            )
        }
    
    async def get_key_pattern(self, reserve_id: str, db: Session) -> Dict[str, Any]:
        """Get current key generation pattern"""
        # Implementation would query tenant database
        return {
            'pattern': settings.KEY_DEFAULT_PATTERN,
            'prefix': '',
            'exclude_similar': True,
            'checksum_enabled': False
        }
    
    async def test_pattern(self, reserve_id: str, count: int, db: Session) -> Dict[str, Any]:
        """Test current key generation pattern"""
        pattern = await self.get_key_pattern(reserve_id, db)
        
        keys = self.key_generator.generate_batch(
            pattern['pattern'],
            count,
            pattern.get('prefix', ''),
            pattern.get('exclude_similar', True),
            pattern.get('checksum_enabled', False)
        )
        
        return {
            'pattern': pattern,
            'sample_keys': keys
        }
    
    async def get_alert_config(self, reserve_id: str, duration: str, db: Session) -> Dict[str, Any]:
        """Get low key alert configuration"""
        # Implementation would query tenant database
        return {
            'duration': duration,
            'threshold': settings.KEY_LOW_STOCK_ALERT,
            'enabled': True,
            'notify_admin': True,
            'notify_owner': False,
            'auto_generate': False
        }
    
    async def count_available_keys(self, reserve_id: str, duration: str, db: Session) -> int:
        """Count available keys for a duration"""
        # Implementation would query tenant database
        return 0
    
    async def delete_batch(self, reserve_id: str, batch_id: str, db: Session) -> Dict[str, Any]:
        """Delete an entire key batch"""
        # Implementation would delete from tenant database
        return {
            'batch_id': batch_id,
            'deleted': True
        }
    
    async def export_keys(self, reserve_id: str, format: str, duration: Optional[str],
                         status: Optional[str], db: Session) -> io.StringIO:
        """Export keys to CSV"""
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow(['Key', 'Type', 'Duration', 'Price', 'Status', 'User ID', 'Created At', 'Expires At'])
        
        # Get keys (implementation would query tenant database)
        # For now, just write header
        
        output.seek(0)
        return output
    
    def _validate_key_format(self, key: str) -> bool:
        """Validate key format"""
        # Basic validation - can be customized
        return len(key) >= settings.KEY_MIN_LENGTH and len(key) <= settings.KEY_MAX_LENGTH
    
    def _validate_pattern(self, pattern: str) -> bool:
        """Validate key pattern"""
        # Check if pattern contains valid variables
        valid_vars = ['{DATE}', '{YYYYMMDD}', '{DDMMYYYY}', '{RANDOM4}', 
                     '{RANDOM6}', '{RANDOM8}', '{RANDOM10}', '{PREFIX}', '{CHECKSUM}']
        
        # Simple validation - can be enhanced
        return len(pattern) > 0
    
    def _count_by_duration(self, keys: List[Dict]) -> Dict[str, int]:
        """Count keys by duration"""
        counts = {}
        for key in keys:
            duration = key['duration']
            counts[duration] = counts.get(duration, 0) + 1
        return counts