# backend/admin_panel/services/tenant_service.py
from typing import Dict, Any, Optional, List
import logging
import asyncpg
from sqlalchemy import create_engine, text
from sqlalchemy.orm import sessionmaker

from core.utils import security
from core.exceptions import DatabaseError

logger = logging.getLogger(__name__)

class TenantService:
    """Service for tenant database management"""
    
    async def create_tenant(self, admin_id: int, reserve_id: str, db) -> bool:
        """Create tenant database for new admin"""
        try:
            # Generate database name
            db_name = f"tenant_{reserve_id.lower()}"
            db_user = f"user_{reserve_id.lower()}"
            db_password = security.generate_api_secret()
            
            # Connect to PostgreSQL as superuser
            conn = await asyncpg.connect(
                host='postgres',
                port=5432,
                user='postgres',
                password='postgres',
                database='postgres'
            )
            
            # Create database
            await conn.execute(f'CREATE DATABASE {db_name}')
            
            # Create user
            await conn.execute(f"CREATE USER {db_user} WITH PASSWORD '{db_password}'")
            
            # Grant privileges
            await conn.execute(f'GRANT ALL PRIVILEGES ON DATABASE {db_name} TO {db_user}')
            
            await conn.close()
            
            # Connect to new database and create tables
            tenant_url = f"postgresql://{db_user}:{db_password}@postgres:5432/{db_name}"
            engine = create_engine(tenant_url)
            
            # Create tables using SQL from schema
            with open('database/schema.sql', 'r') as f:
                schema_sql = f.read()
            
            with engine.connect() as conn:
                # Replace template table names with actual names
                schema_sql = schema_sql.replace('tenant_users_template', f'users_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_keys_template', f'keys_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_transactions_template', f'transactions_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_key_generation_rules_template', f'key_rules_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_key_batches_template', f'key_batches_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_apk_versions_template', f'apk_versions_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_broadcasts_template', f'broadcasts_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_broadcast_recipients_template', f'broadcast_recipients_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_user_activity_log_template', f'user_activity_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_user_sessions_template', f'user_sessions_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_referrals_template', f'referrals_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_support_tickets_template', f'support_tickets_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_ticket_replies_template', f'ticket_replies_{reserve_id.lower()}')
                schema_sql = schema_sql.replace('tenant_key_alert_config_template', f'key_alerts_{reserve_id.lower()}')
                
                # Execute schema
                conn.execute(text(schema_sql))
                conn.commit()
            
            logger.info(f"Tenant database created for {reserve_id}")
            return True
            
        except Exception as e:
            logger.error(f"Error creating tenant database: {e}")
            raise DatabaseError(f"Failed to create tenant database: {str(e)}")
    
    async def delete_tenant(self, reserve_id: str) -> bool:
        """Delete tenant database"""
        try:
            db_name = f"tenant_{reserve_id.lower()}"
            db_user = f"user_{reserve_id.lower()}"
            
            conn = await asyncpg.connect(
                host='postgres',
                port=5432,
                user='postgres',
                password='postgres',
                database='postgres'
            )
            
            # Drop database
            await conn.execute(f'DROP DATABASE IF EXISTS {db_name}')
            
            # Drop user
            await conn.execute(f'DROP USER IF EXISTS {db_user}')
            
            await conn.close()
            
            logger.info(f"Tenant database deleted for {reserve_id}")
            return True
            
        except Exception as e:
            logger.error(f"Error deleting tenant database: {e}")
            return False
    
    async def get_stats(self, reserve_id: str) -> Dict[str, Any]:
        """Get tenant statistics"""
        # This would query tenant database
        return {
            'total_users': 5678,
            'active_today': 456,
            'total_keys': 15000,
            'available_keys': 5000,
            'sold_keys': 10000,
            'total_revenue': 6000000,
            'monthly_revenue': 750000
        }
    
    async def get_basic_stats(self, reserve_id: str) -> Dict[str, Any]:
        """Get basic tenant statistics"""
        return {
            'total_users': 5678,
            'total_revenue': 6000000
        }
    
    async def get_users(self, reserve_id: str, search: Optional[str], 
                       skip: int, limit: int) -> List[Dict[str, Any]]:
        """Get users from tenant database"""
        # This would query tenant database
        return []
    
    async def get_user(self, reserve_id: str, telegram_id: int) -> Optional[Dict[str, Any]]:
        """Get user from tenant database"""
        # This would query tenant database
        return None
    
    async def get_user_activity(self, reserve_id: str, telegram_id: int, 
                               limit: int) -> List[Dict[str, Any]]:
        """Get user activity from tenant database"""
        # This would query tenant database
        return []
    
    async def get_user_keys(self, reserve_id: str, telegram_id: int) -> List[Dict[str, Any]]:
        """Get user keys from tenant database"""
        # This would query tenant database
        return []
    
    async def get_activities(self, reserve_id: str, user_id: Optional[int],
                            activity_type: Optional[str], date_from: Optional[date],
                            date_to: Optional[date], skip: int, limit: int) -> List[Dict[str, Any]]:
        """Get activities from tenant database"""
        # This would query tenant database
        return []
    
    async def get_activity_stats(self, reserve_id: str, date_from: date,
                                date_to: date) -> Dict[str, Any]:
        """Get activity statistics from tenant database"""
        # This would query tenant database
        return {}
    
    async def get_activity_count(self, reserve_id: str, date_from: date,
                                date_to: date) -> int:
        """Get activity count from tenant database"""
        # This would query tenant database
        return 0
    
    async def get_all_activities(self, reserve_id: str, date_from: Optional[date],
                                date_to: Optional[date]) -> List[Dict[str, Any]]:
        """Get all activities from tenant database"""
        # This would query tenant database
        return []
    
    async def get_recent_activities(self, reserve_id: str, seconds: int,
                                   limit: int) -> List[Dict[str, Any]]:
        """Get recent activities from tenant database"""
        # This would query tenant database
        return []
    
    async def block_user(self, reserve_id: str, telegram_id: int, reason: str):
        """Block user in tenant database"""
        # This would update tenant database
        pass
    
    async def unblock_user(self, reserve_id: str, telegram_id: int):
        """Unblock user in tenant database"""
        # This would update tenant database
        pass
    
    async def get_all_users(self, reserve_id: str) -> List[Dict[str, Any]]:
        """Get all users from tenant database"""
        # This would query tenant database
        return []