# backend/api/dependencies/auth_deps.py
from fastapi import Depends, HTTPException, status
from fastapi.security import OAuth2PasswordBearer
from typing import Optional
from sqlalchemy.orm import Session

from core.database import get_db
from core.security import get_current_user, get_current_owner, get_current_admin
from core.models import Owner, Admin

oauth2_scheme = OAuth2PasswordBearer(tokenUrl="/api/v1/auth/login", auto_error=False)

async def optional_current_user(
    token: Optional[str] = Depends(oauth2_scheme),
    db: Session = Depends(get_db)
):
    """Get current user if authenticated, otherwise None"""
    if not token:
        return None
    try:
        return await get_current_user(token, db)
    except:
        return None


async def require_active_admin(
    current_admin: Admin = Depends(get_current_admin)
) -> Admin:
    """Require active admin with valid recharge"""
    if current_admin.recharge_status != 'active':
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Admin panel expired. Please recharge."
        )
    if current_admin.status != 'active':
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Admin account is not active"
        )
    return current_admin


async def require_owner_with_2fa(
    current_owner: Owner = Depends(get_current_owner)
) -> Owner:
    """Require owner with 2FA if enabled"""
    # Check if 2FA is required but not verified in this session
    # Implementation would check session for 2FA verification
    return current_owner


class PermissionChecker:
    """Permission checking dependency"""
    
    def __init__(self, required_permissions: list):
        self.required_permissions = required_permissions
    
    async def __call__(self, current_user = Depends(get_current_user)):
        """Check if user has required permissions"""
        # Implementation would check user permissions
        # For now, just return the user
        return current_user


def has_permission(permission: str):
    """Dependency to check specific permission"""
    return PermissionChecker([permission])