# backend/api/dependencies/db_deps.py
from typing import Optional, Generator
from sqlalchemy.orm import Session
from fastapi import Request, Depends, HTTPException

from core.database import SessionLocal, tenant_db_manager
from core.models import TenantRegistry, Admin
from core.security import get_current_admin


def get_db() -> Generator[Session, None, None]:
    """Get database session"""
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()


async def get_tenant_db(
    request: Request,
    current_admin: Admin = Depends(get_current_admin),
    db: Session = Depends(get_db)
) -> Session:
    """Get tenant database session for current admin"""
    # Get tenant registry
    tenant = db.query(TenantRegistry).filter(
        TenantRegistry.admin_id == current_admin.id
    ).first()
    
    if not tenant:
        raise HTTPException(status_code=404, detail="Tenant not found")
    
    # Create database URL
    db_url = f"postgresql://{tenant.database_user}:{tenant.database_password}@{tenant.database_host}:{tenant.database_port}/{tenant.database_name}"
    
    # Get or create tenant session
    tenant_session = tenant_db_manager.get_tenant_session(
        current_admin.reserve_id, db_url
    )
    
    # Store in request state for cleanup
    request.state.tenant_session = tenant_session
    
    return tenant_session


async def get_tenant_by_reserve(
    reserve_id: str,
    db: Session = Depends(get_db)
) -> Session:
    """Get tenant database session by reserve ID"""
    tenant = db.query(TenantRegistry).filter(
        TenantRegistry.reserve_id == reserve_id
    ).first()
    
    if not tenant:
        raise HTTPException(status_code=404, detail="Tenant not found")
    
    db_url = f"postgresql://{tenant.database_user}:{tenant.database_password}@{tenant.database_host}:{tenant.database_port}/{tenant.database_name}"
    
    return tenant_db_manager.get_tenant_session(reserve_id, db_url)


async def get_tenant_registry(
    reserve_id: str,
    db: Session = Depends(get_db)
) -> TenantRegistry:
    """Get tenant registry entry"""
    tenant = db.query(TenantRegistry).filter(
        TenantRegistry.reserve_id == reserve_id
    ).first()
    
    if not tenant:
        raise HTTPException(status_code=404, detail="Tenant not found")
    
    return tenant


def close_tenant_db(request: Request):
    """Close tenant database session"""
    if hasattr(request.state, 'tenant_session'):
        request.state.tenant_session.close()