# backend/api/tasks/celery_app.py
from celery import Celery
from celery.schedules import crontab
import logging
from typing import Any

from core.config import settings

logger = logging.getLogger(__name__)

# Create Celery app
celery_app = Celery(
    "telegram_bot",
    broker=settings.CELERY_BROKER_URL,
    backend=settings.CELERY_RESULT_BACKEND,
    include=[
        "api.tasks.broadcast_tasks",
        "api.tasks.key_tasks",
        "api.tasks.payment_tasks",
        "api.tasks.monitoring_tasks",
        "api.tasks.notification_tasks",
        "api.tasks.report_tasks",
        "api.tasks.backup_tasks",
    ]
)

# Configure Celery
celery_app.conf.update(
    task_serializer="json",
    result_serializer="json",
    accept_content=["json"],
    timezone="UTC",
    enable_utc=True,
    task_track_started=True,
    task_time_limit=30 * 60,  # 30 minutes
    task_soft_time_limit=25 * 60,  # 25 minutes
    result_expires=3600,
    worker_prefetch_multiplier=1,
    worker_max_tasks_per_child=1000,
    task_acks_late=True,
    task_reject_on_worker_lost=True,
)

# Scheduled tasks
celery_app.conf.beat_schedule = {
    # Monitoring tasks
    "check-server-health": {
        "task": "api.tasks.monitoring_tasks.check_all_servers_health",
        "schedule": crontab(minute="*/5"),  # Every 5 minutes
    },
    "check-expiring-recharges": {
        "task": "api.tasks.monitoring_tasks.check_expiring_recharges",
        "schedule": crontab(hour="0", minute="0"),  # Daily at midnight
    },
    "check-low-keys": {
        "task": "api.tasks.key_tasks.check_low_keys_all_tenants",
        "schedule": crontab(minute="*/30"),  # Every 30 minutes
    },
    
    # Backup tasks
    "daily-database-backup": {
        "task": "api.tasks.backup_tasks.perform_database_backup",
        "schedule": crontab(hour="2", minute="0"),  # Daily at 2 AM
    },
    
    # Report tasks
    "send-daily-reports": {
        "task": "api.tasks.report_tasks.send_daily_reports",
        "schedule": crontab(hour="8", minute="0"),  # Daily at 8 AM
    },
    "send-weekly-reports": {
        "task": "api.tasks.report_tasks.send_weekly_reports",
        "schedule": crontab(day_of_week="monday", hour="9", minute="0"),  # Monday at 9 AM
    },
    
    # Cleanup tasks
    "cleanup-old-activities": {
        "task": "api.tasks.monitoring_tasks.cleanup_old_activities",
        "schedule": crontab(hour="3", minute="0"),  # Daily at 3 AM
    },
    "cleanup-expired-keys": {
        "task": "api.tasks.key_tasks.cleanup_expired_keys",
        "schedule": crontab(hour="4", minute="0"),  # Daily at 4 AM
    },
    
    # Notification tasks
    "send-renewal-reminders": {
        "task": "api.tasks.notification_tasks.send_renewal_reminders",
        "schedule": crontab(hour="10", minute="0"),  # Daily at 10 AM
    },
    
    # Analytics tasks
    "update-analytics": {
        "task": "api.tasks.report_tasks.update_analytics",
        "schedule": crontab(minute="*/15"),  # Every 15 minutes
    },
}


@celery_app.task(bind=True, max_retries=3)
def debug_task(self: Any) -> dict:
    """Debug task to test Celery"""
    logger.info(f"Debug task executed: {self.request.id}")
    return {
        "status": "success",
        "task_id": self.request.id,
        "message": "Debug task completed"
    }


@celery_app.task(bind=True)
def health_check(self: Any) -> dict:
    """Health check task"""
    return {
        "status": "healthy",
        "task_id": self.request.id,
        "timestamp": str(celery_app.now())
    }


# Task base class with common functionality
class BaseTaskWithRetry(celery_app.Task):
    """Base task class with retry and error handling"""
    
    autoretry_for = (Exception,)
    retry_kwargs = {"max_retries": 3}
    retry_backoff = True
    retry_backoff_max = 600
    retry_jitter = True
    
    def on_failure(self, exc, task_id, args, kwargs, einfo):
        """Handle task failure"""
        logger.error(f"Task {task_id} failed: {exc}")
        super().on_failure(exc, task_id, args, kwargs, einfo)


# Set custom task base
celery_app.Task = BaseTaskWithRetry