# backend/api/tasks/key_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
from typing import Dict, Any, List

from core.database import SessionLocal
from core.models import Admin, TenantRegistry
from admin_panel.services.key_service import KeyService
from core.utils.datetime_utils import datetime_utils
from core.config import settings

logger = get_task_logger(__name__)


@shared_task
def check_low_keys_all_tenants():
    """Check low key alerts for all tenants"""
    logger.info("Checking low keys for all tenants...")
    
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(
            Admin.recharge_status == 'active',
            Admin.status == 'active'
        ).all()
        
        results = []
        for admin in admins:
            result = check_tenant_low_keys.delay(admin.id, admin.reserve_id)
            results.append(result.id)
        
        return {
            "tenants_checked": len(admins),
            "task_ids": results,
            "timestamp": datetime_utils.now().isoformat()
        }
    finally:
        db.close()


@shared_task
def check_tenant_low_keys(admin_id: int, reserve_id: str):
    """Check low keys for specific tenant"""
    logger.info(f"Checking low keys for tenant {reserve_id}")
    
    try:
        db = SessionLocal()
        try:
            tenant = db.query(TenantRegistry).filter(
                TenantRegistry.reserve_id == reserve_id
            ).first()
            
            if not tenant:
                logger.error(f"Tenant {reserve_id} not found")
                return {"error": "Tenant not found", "reserve_id": reserve_id}
            
            key_service = KeyService()
            
            # Check each duration
            durations = ['1d', '3d', '7d', '30d', '60d']
            alerts = []
            
            for duration in durations:
                # Get alert config
                threshold = settings.KEY_LOW_STOCK_ALERT
                
                # Count available keys
                # This would query tenant database
                count = 500  # Mock value
                
                if count <= threshold:
                    alerts.append({
                        'duration': duration,
                        'count': count,
                        'threshold': threshold
                    })
                    
                    # Send notification
                    send_low_key_notification.delay(
                        admin_id, reserve_id, duration, count, threshold
                    )
            
            return {
                "tenant": reserve_id,
                "admin_id": admin_id,
                "checked_at": datetime_utils.now().isoformat(),
                "alerts": alerts
            }
        finally:
            db.close()
            
    except Exception as e:
        logger.error(f"Error checking low keys for {reserve_id}: {e}")
        return {"error": str(e), "reserve_id": reserve_id}


@shared_task
def send_low_key_notification(admin_id: int, reserve_id: str, 
                              duration: str, count: int, threshold: int):
    """Send low key alert notification to admin"""
    logger.info(f"Sending low key alert to admin {admin_id} for {duration} keys")
    
    message = (
        f"⚠️ **Low Key Alert**\n\n"
        f"**Bot:** {reserve_id}\n"
        f"**Duration:** {duration}\n"
        f"**Current Count:** {count}\n"
        f"**Threshold:** {threshold}\n\n"
        f"Please add more keys to avoid disruption."
    )
    
    # In production, this would send a Telegram message
    # from bot_core.bot import bot_manager
    # bot = bot_manager.get_bot(admin_id)
    # if bot:
    #     await bot.application.bot.send_message(admin_id, message)
    
    return {
        "admin_id": admin_id,
        "reserve_id": reserve_id,
        "duration": duration,
        "count": count,
        "threshold": threshold,
        "notification_sent": True,
        "message": message
    }


@shared_task
def cleanup_expired_keys():
    """Mark expired keys as expired"""
    logger.info("Cleaning up expired keys...")
    
    # This would query all tenant databases and mark expired keys
    total_expired = 0
    
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(Admin.recharge_status == 'active').all()
        
        for admin in admins:
            # This would update tenant database
            # expired = mark_expired_keys_in_tenant(admin.reserve_id)
            # total_expired += expired
            pass
            
    finally:
        db.close()
    
    logger.info(f"Marked {total_expired} keys as expired")
    return {"expired_keys_marked": total_expired}


@shared_task
def generate_key_batch(admin_id: int, reserve_id: str, duration: str, 
                       count: int, pattern: Dict[str, Any]):
    """Generate batch of keys"""
    logger.info(f"Generating {count} keys for {duration} for tenant {reserve_id}")
    
    try:
        from core.utils.key_generator import KeyGenerator
        key_gen = KeyGenerator()
        
        keys = key_gen.generate_batch(
            pattern=pattern.get('pattern', '{PREFIX}-{YYYYMMDD}-{RANDOM8}'),
            count=count,
            prefix=pattern.get('prefix', ''),
            exclude_similar=pattern.get('exclude_similar', True),
            enable_checksum=pattern.get('checksum_enabled', False)
        )
        
        # This would save to tenant database
        
        return {
            "tenant": reserve_id,
            "admin_id": admin_id,
            "duration": duration,
            "requested": count,
            "generated": len(keys),
            "sample_keys": keys[:5],
            "timestamp": datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Key generation failed: {e}")
        return {"error": str(e), "tenant": reserve_id}