# backend/api/tasks/monitoring_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
import logging
import psutil
import platform
import os
import socket

from core.database import SessionLocal
from core.models import Admin, ServerHealthLog, ServerIncident, Owner
from core.utils.datetime_utils import datetime_utils
from core.config import settings

logger = get_task_logger(__name__)


@shared_task
def check_all_servers_health():
    """Check health of all servers"""
    logger.info("Checking all servers health...")
    
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(
            Admin.status == 'active'
        ).all()
        
        results = []
        for admin in admins:
            result = check_single_server_health.delay(admin.id, admin.reserve_id)
            results.append({
                'admin_id': admin.id,
                'reserve_id': admin.reserve_id,
                'task_id': result.id
            })
        
        logger.info(f"Health checks initiated for {len(admins)} servers")
        return {
            'servers_checked': len(admins),
            'task_ids': [r['task_id'] for r in results],
            'details': results,
            'timestamp': datetime_utils.now().isoformat()
        }
    finally:
        db.close()


@shared_task
def check_single_server_health(admin_id: int, reserve_id: str):
    """Check health of a single server"""
    logger.info(f"Checking health for server {reserve_id}")
    
    try:
        # Get system metrics
        cpu_percent = psutil.cpu_percent(interval=1)
        cpu_count = psutil.cpu_count()
        cpu_freq = psutil.cpu_freq()
        
        memory = psutil.virtual_memory()
        swap = psutil.swap_memory()
        
        disk = psutil.disk_usage('/')
        disk_io = psutil.disk_io_counters()
        
        # Network stats
        net_io = psutil.net_io_counters()
        net_connections = len(psutil.net_connections())
        net_if_stats = psutil.net_if_stats()
        
        # Get hostname and IP
        hostname = socket.gethostname()
        try:
            ip_address = socket.gethostbyname(hostname)
        except:
            ip_address = '127.0.0.1'
        
        # Check service status
        bot_status = 'online'
        webhook_status = 'online'
        database_status = 'online'
        redis_status = 'online'
        
        # Check if services are responding
        # This would actually ping the services
        
        # Determine overall status
        if cpu_percent > 90:
            webhook_status = 'degraded'
        if memory.percent > 90:
            database_status = 'degraded'
        if disk.percent > 95:
            redis_status = 'degraded'
        
        # Calculate response time (mock)
        response_time = 150 + (cpu_percent // 10) * 5  # Mock: increases with CPU
        
        # Create health log
        db = SessionLocal()
        try:
            health_log = ServerHealthLog(
                admin_id=admin_id,
                reserve_id=reserve_id,
                server_name=hostname,
                server_ip=ip_address,
                cpu_usage=cpu_percent,
                ram_usage=memory.percent,
                disk_usage=disk.percent,
                network_in=net_io.bytes_recv / 1024 / 1024,  # MB
                network_out=net_io.bytes_sent / 1024 / 1024,  # MB
                bot_status=bot_status,
                webhook_status=webhook_status,
                database_status=database_status,
                redis_status=redis_status,
                response_time=response_time,
                checked_at=datetime_utils.now()
            )
            db.add(health_log)
            db.commit()
            
            # Determine overall health status
            is_healthy = all([
                cpu_percent < 80,
                memory.percent < 80,
                disk.percent < 80,
                bot_status == 'online',
                webhook_status == 'online',
                database_status == 'online',
                redis_status == 'online'
            ])
            
            health_status = 'healthy' if is_healthy else 'degraded'
            
            # Check for alerts
            alerts = []
            
            if cpu_percent > 80:
                alert = create_incident.delay(
                    admin_id, reserve_id, 'high_cpu',
                    f"CPU usage at {cpu_percent}% (threshold: 80%)", 
                    'warning' if cpu_percent < 90 else 'critical'
                )
                alerts.append(alert.id)
            
            if memory.percent > 85:
                alert = create_incident.delay(
                    admin_id, reserve_id, 'high_memory',
                    f"Memory usage at {memory.percent}% (threshold: 85%)",
                    'warning' if memory.percent < 95 else 'critical'
                )
                alerts.append(alert.id)
            
            if disk.percent > 90:
                severity = 'critical' if disk.percent > 95 else 'warning'
                alert = create_incident.delay(
                    admin_id, reserve_id, 'low_disk',
                    f"Disk usage at {disk.percent}% (threshold: 90%)",
                    severity
                )
                alerts.append(alert.id)
            
            if bot_status != 'online':
                alert = create_incident.delay(
                    admin_id, reserve_id, 'bot_offline',
                    f"Bot is {bot_status}",
                    'critical'
                )
                alerts.append(alert.id)
            
            logger.info(
                f"Health check completed for {reserve_id}: "
                f"CPU={cpu_percent}%, MEM={memory.percent}%, "
                f"DISK={disk.percent}%, STATUS={health_status}"
            )
            
            return {
                'reserve_id': reserve_id,
                'admin_id': admin_id,
                'hostname': hostname,
                'ip_address': ip_address,
                'cpu': {
                    'percent': cpu_percent,
                    'count': cpu_count,
                    'frequency': cpu_freq.current if cpu_freq else None
                },
                'memory': {
                    'percent': memory.percent,
                    'total': memory.total,
                    'available': memory.available,
                    'used': memory.used,
                    'swap_percent': swap.percent
                },
                'disk': {
                    'percent': disk.percent,
                    'total': disk.total,
                    'used': disk.used,
                    'free': disk.free
                },
                'network': {
                    'bytes_sent_mb': net_io.bytes_sent / 1024 / 1024,
                    'bytes_recv_mb': net_io.bytes_recv / 1024 / 1024,
                    'connections': net_connections
                },
                'services': {
                    'bot': bot_status,
                    'webhook': webhook_status,
                    'database': database_status,
                    'redis': redis_status
                },
                'response_time_ms': response_time,
                'status': health_status,
                'alerts_triggered': len(alerts),
                'alert_ids': alerts,
                'checked_at': datetime_utils.now().isoformat()
            }
            
        finally:
            db.close()
            
    except Exception as e:
        logger.error(f"Health check failed for {reserve_id}: {e}")
        return {
            'reserve_id': reserve_id,
            'error': str(e),
            'status': 'failed',
            'checked_at': datetime_utils.now().isoformat()
        }


@shared_task
def create_incident(admin_id: int, reserve_id: str, incident_type: str,
                   description: str, severity: str):
    """Create server incident record"""
    logger.info(f"Creating incident for {reserve_id}: {incident_type} ({severity})")
    
    db = SessionLocal()
    try:
        # Check if similar incident already exists and is still open
        existing = db.query(ServerIncident).filter(
            ServerIncident.reserve_id == reserve_id,
            ServerIncident.incident_type == incident_type,
            ServerIncident.status == 'open'
        ).first()
        
        if existing:
            logger.info(f"Similar incident already exists for {reserve_id}: {existing.id}")
            return {
                'incident_id': existing.id,
                'reserve_id': reserve_id,
                'status': 'already_exists'
            }
        
        incident = ServerIncident(
            admin_id=admin_id,
            reserve_id=reserve_id,
            incident_type=incident_type,
            severity=severity,
            start_time=datetime_utils.now(),
            description=description,
            status='open'
        )
        db.add(incident)
        db.commit()
        db.refresh(incident)
        
        # Send notification
        send_incident_notification.delay(incident.id)
        
        logger.info(f"Incident {incident.id} created for {reserve_id}")
        
        return {
            'incident_id': incident.id,
            'reserve_id': reserve_id,
            'type': incident_type,
            'severity': severity,
            'description': description,
            'status': 'open',
            'created_at': incident.start_time.isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to create incident: {e}")
        db.rollback()
        return {
            'error': str(e),
            'reserve_id': reserve_id
        }
    finally:
        db.close()


@shared_task
def check_expiring_recharges():
    """Check for expiring admin recharges"""
    logger.info("Checking expiring recharges...")
    
    now = datetime_utils.now()
    warning_periods = [7, 3, 1]  # Days before expiry
    results = []
    
    db = SessionLocal()
    try:
        for days in warning_periods:
            expiry_date = now + timedelta(days=days)
            start_range = expiry_date - timedelta(days=1)
            end_range = expiry_date + timedelta(days=1)
            
            admins = db.query(Admin).filter(
                Admin.recharge_expiry.between(start_range, end_range),
                Admin.recharge_status == 'active'
            ).all()
            
            for admin in admins:
                result = send_expiry_notification.delay(
                    admin.id, 
                    admin.admin_username, 
                    admin.bot_name,
                    days,
                    admin.recharge_expiry.isoformat() if admin.recharge_expiry else None
                )
                results.append({
                    'admin_id': admin.id,
                    'username': admin.admin_username,
                    'bot_name': admin.bot_name,
                    'days_left': days,
                    'expiry_date': admin.recharge_expiry.isoformat() if admin.recharge_expiry else None,
                    'task_id': result.id
                })
                
                logger.info(f"Expiry warning initiated for admin {admin.id} ({days} days)")
        
        # Check for already expired admins
        expired_admins = db.query(Admin).filter(
            Admin.recharge_expiry < now,
            Admin.recharge_status == 'active'
        ).all()
        
        for admin in expired_admins:
            # Auto-expire
            admin.recharge_status = 'expired'
            admin.status = 'inactive'
            
            # Send expiration notification
            send_expiry_notification.delay(
                admin.id,
                admin.admin_username,
                admin.bot_name,
                0,
                admin.recharge_expiry.isoformat() if admin.recharge_expiry else None
            )
            
            results.append({
                'admin_id': admin.id,
                'username': admin.admin_username,
                'status': 'expired',
                'expiry_date': admin.recharge_expiry.isoformat() if admin.recharge_expiry else None
            })
            
            logger.info(f"Admin {admin.id} auto-expired")
        
        db.commit()
        
        return {
            'checked_at': now.isoformat(),
            'total_admins_checked': db.query(Admin).count(),
            'expiring_soon': len([r for r in results if r.get('days_left', 0) > 0]),
            'expired_today': len([r for r in results if r.get('status') == 'expired']),
            'details': results,
            'timestamp': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Error checking expiring recharges: {e}")
        return {
            'error': str(e),
            'timestamp': datetime_utils.now().isoformat()
        }
    finally:
        db.close()


@shared_task
def send_expiry_notification(admin_id: int, username: str, bot_name: str, 
                            days_left: int, expiry_date: str = None):
    """Send expiry notification to admin"""
    logger.info(f"Sending expiry notification to admin {admin_id} ({days_left} days)")
    
    # Create message based on days left
    if days_left == 0:
        subject = "❌ Panel Expired"
        message = (
            f"❌ **Panel Expired**\n\n"
            f"Hello {username},\n\n"
            f"Your admin panel for **{bot_name}** has expired.\n"
            f"Expiry Date: {expiry_date}\n\n"
            f"To reactivate, please contact the owner for renewal.\n\n"
            f"Your bot is now in maintenance mode."
        )
        notification_type = 'expired'
    elif days_left == 1:
        subject = "🔴 FINAL WARNING: Expires Tomorrow"
        message = (
            f"🔴 **URGENT: Final Warning**\n\n"
            f"Hello {username},\n\n"
            f"Your admin panel for **{bot_name}** will expire **TOMORROW**!\n"
            f"Expiry Date: {expiry_date}\n\n"
            f"Renew immediately to avoid service interruption.\n"
            f"Contact the owner for renewal."
        )
        notification_type = 'final_warning'
    elif days_left == 3:
        subject = "⚠️ URGENT: 3 Days Left"
        message = (
            f"⚠️ **URGENT: Renewal Required**\n\n"
            f"Hello {username},\n\n"
            f"Your admin panel for **{bot_name}** will expire in **3 days**!\n"
            f"Expiry Date: {expiry_date}\n\n"
            f"Please renew now to maintain service.\n"
            f"Contact the owner for renewal."
        )
        notification_type = 'urgent'
    elif days_left == 7:
        subject = "🔔 Renewal Reminder - 7 Days Left"
        message = (
            f"🔔 **Renewal Reminder**\n\n"
            f"Hello {username},\n\n"
            f"Your admin panel for **{bot_name}** will expire in **7 days**.\n"
            f"Expiry Date: {expiry_date}\n\n"
            f"Please renew to avoid service interruption.\n"
            f"Contact the owner for renewal."
        )
        notification_type = 'reminder'
    else:
        subject = f"Renewal Reminder - {days_left} Days Left"
        message = (
            f"Hello {username},\n\n"
            f"Your admin panel for **{bot_name}** will expire in {days_left} days.\n"
            f"Expiry Date: {expiry_date}\n\n"
            f"Please renew to avoid service interruption."
        )
        notification_type = 'reminder'
    
    db = SessionLocal()
    try:
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if admin and admin.telegram_id:
            # Send Telegram notification
            from .notification_tasks import send_telegram_notification
            send_telegram_notification.delay(
                admin.telegram_id,
                message,
                'HTML'
            )
            
            # Log notification
            from core.models import NotificationLog
            notification = NotificationLog(
                admin_id=admin_id,
                notification_type=f'renewal_{notification_type}',
                channel='telegram',
                recipient_type='admin',
                recipient_id=admin.telegram_id,
                subject=subject,
                message=message,
                status='sent',
                sent_at=datetime_utils.now(),
                metadata={'days_left': days_left, 'expiry_date': expiry_date}
            )
            db.add(notification)
            db.commit()
            
            logger.info(f"Expiry notification sent to admin {admin_id} ({days_left} days)")
            
            return {
                'admin_id': admin_id,
                'username': username,
                'days_left': days_left,
                'notification_type': notification_type,
                'notification_sent': True,
                'sent_at': datetime_utils.now().isoformat()
            }
        
        return {
            'admin_id': admin_id,
            'username': username,
            'days_left': days_left,
            'notification_sent': False,
            'reason': 'Admin not found or no telegram_id'
        }
        
    except Exception as e:
        logger.error(f"Failed to send expiry notification: {e}")
        return {
            'admin_id': admin_id,
            'error': str(e),
            'notification_sent': False
        }
    finally:
        db.close()


@shared_task
def send_incident_notification(incident_id: int):
    """Send incident notification"""
    logger.info(f"Sending incident notification for incident {incident_id}")
    
    db = SessionLocal()
    try:
        incident = db.query(ServerIncident).filter(
            ServerIncident.id == incident_id
        ).first()
        
        if not incident:
            logger.error(f"Incident {incident_id} not found")
            return {'error': 'Incident not found'}
        
        # Get admin and owner
        admin = db.query(Admin).filter(Admin.id == incident.admin_id).first()
        owner = db.query(Owner).first()
        
        # Create message
        severity_emoji = {
            'info': 'ℹ️',
            'warning': '⚠️',
            'critical': '🔴',
            'emergency': '🚨'
        }.get(incident.severity, 'ℹ️')
        
        severity_color = {
            'info': 'blue',
            'warning': 'orange',
            'critical': 'red',
            'emergency': 'darkred'
        }.get(incident.severity, 'blue')
        
        message = (
            f"{severity_emoji} **Server Incident - {incident.severity.upper()}**\n\n"
            f"**Bot:** {incident.reserve_id}\n"
            f"**Type:** {incident.incident_type.replace('_', ' ').title()}\n"
            f"**Severity:** {incident.severity.upper()}\n"
            f"**Description:** {incident.description}\n"
            f"**Time:** {incident.start_time.strftime('%Y-%m-%d %H:%M:%S UTC')}\n\n"
            f"**Action Required:** Please investigate immediately."
        )
        
        notifications_sent = []
        
        # Send to admin
        if admin and admin.telegram_id:
            from .notification_tasks import send_telegram_notification
            send_telegram_notification.delay(admin.telegram_id, message, 'HTML')
            notifications_sent.append('admin')
            
            logger.info(f"Incident notification sent to admin {admin.id}")
        
        # Send to owner
        if owner and owner.telegram_id:
            from .notification_tasks import send_telegram_notification
            send_telegram_notification.delay(owner.telegram_id, message, 'HTML')
            notifications_sent.append('owner')
            
            logger.info(f"Incident notification sent to owner")
        
        # Update incident with notification info
        incident.metadata = incident.metadata or {}
        incident.metadata['notifications_sent'] = notifications_sent
        incident.metadata['notified_at'] = datetime_utils.now().isoformat()
        db.commit()
        
        return {
            'incident_id': incident_id,
            'reserve_id': incident.reserve_id,
            'severity': incident.severity,
            'notifications_sent': notifications_sent,
            'sent_at': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to send incident notification: {e}")
        return {
            'incident_id': incident_id,
            'error': str(e)
        }
    finally:
        db.close()


@shared_task
def resolve_incident(incident_id: int, resolution: str):
    """Resolve an incident"""
    logger.info(f"Resolving incident {incident_id}")
    
    db = SessionLocal()
    try:
        incident = db.query(ServerIncident).filter(
            ServerIncident.id == incident_id
        ).first()
        
        if not incident:
            return {'error': 'Incident not found'}
        
        incident.status = 'resolved'
        incident.end_time = datetime_utils.now()
        incident.resolution = resolution
        incident.metadata = incident.metadata or {}
        incident.metadata['resolved_at'] = datetime_utils.now().isoformat()
        
        db.commit()
        
        logger.info(f"Incident {incident_id} resolved")
        
        return {
            'incident_id': incident_id,
            'status': 'resolved',
            'resolved_at': datetime_utils.now().isoformat(),
            'resolution': resolution
        }
        
    except Exception as e:
        logger.error(f"Failed to resolve incident: {e}")
        return {
            'incident_id': incident_id,
            'error': str(e)
        }
    finally:
        db.close()


@shared_task
def cleanup_old_activities():
    """Clean up old activity logs"""
    logger.info("Cleaning up old activities...")
    
    retention_days = settings.ACTIVITY_RETENTION_DAYS
    cutoff = datetime_utils.now() - timedelta(days=retention_days)
    
    # This would delete old activities from all tenant databases
    # For now, just log
    
    logger.info(f"Activities older than {cutoff.strftime('%Y-%m-%d')} cleaned up")
    
    return {
        'cleaned_up_to': cutoff.isoformat(),
        'retention_days': retention_days,
        'status': 'completed',
        'timestamp': datetime_utils.now().isoformat()
    }


@shared_task
def cleanup_old_health_logs():
    """Clean up old health logs"""
    logger.info("Cleaning up old health logs...")
    
    retention_days = 30  # Keep health logs for 30 days
    cutoff = datetime_utils.now() - timedelta(days=retention_days)
    
    db = SessionLocal()
    try:
        result = db.query(ServerHealthLog).filter(
            ServerHealthLog.checked_at < cutoff
        ).delete()
        
        db.commit()
        
        logger.info(f"Cleaned up {result} old health logs")
        
        return {
            'deleted_count': result,
            'retention_days': retention_days,
            'cutoff_date': cutoff.isoformat(),
            'timestamp': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to clean up health logs: {e}")
        return {
            'error': str(e)
        }
    finally:
        db.close()


@shared_task
def check_system_resources():
    """Check overall system resources"""
    logger.info("Checking system resources...")
    
    try:
        # CPU
        cpu_percent = psutil.cpu_percent(interval=1)
        cpu_count = psutil.cpu_count()
        
        # Memory
        memory = psutil.virtual_memory()
        
        # Disk
        disk = psutil.disk_usage('/')
        
        # Network
        net_io = psutil.net_io_counters()
        
        # Processes
        process_count = len(psutil.pids())
        
        # Uptime
        boot_time = datetime.fromtimestamp(psutil.boot_time())
        uptime = datetime_utils.now() - boot_time
        
        status = 'healthy'
        alerts = []
        
        if cpu_percent > 80:
            status = 'degraded'
            alerts.append(f"High CPU: {cpu_percent}%")
        
        if memory.percent > 85:
            status = 'degraded'
            alerts.append(f"High Memory: {memory.percent}%")
        
        if disk.percent > 90:
            status = 'degraded'
            alerts.append(f"Low Disk: {disk.percent}%")
        
        logger.info(f"System resources check completed: CPU={cpu_percent}%, MEM={memory.percent}%, DISK={disk.percent}%")
        
        return {
            'status': status,
            'alerts': alerts,
            'cpu': {
                'percent': cpu_percent,
                'count': cpu_count
            },
            'memory': {
                'total_gb': memory.total / (1024**3),
                'available_gb': memory.available / (1024**3),
                'percent': memory.percent
            },
            'disk': {
                'total_gb': disk.total / (1024**3),
                'free_gb': disk.free / (1024**3),
                'percent': disk.percent
            },
            'network': {
                'bytes_sent_mb': net_io.bytes_sent / (1024**2),
                'bytes_recv_mb': net_io.bytes_recv / (1024**2)
            },
            'processes': process_count,
            'uptime': {
                'seconds': uptime.total_seconds(),
                'days': uptime.days,
                'hours': uptime.seconds // 3600
            },
            'boot_time': boot_time.isoformat(),
            'checked_at': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"System resources check failed: {e}")
        return {
            'error': str(e),
            'status': 'failed',
            'checked_at': datetime_utils.now().isoformat()
        }


@shared_task
def monitor_bot_performance():
    """Monitor bot performance metrics"""
    logger.info("Monitoring bot performance...")
    
    # This would collect metrics from all running bots
    # For now, return mock data
    
    db = SessionLocal()
    try:
        active_admins = db.query(Admin).filter(
            Admin.recharge_status == 'active',
            Admin.status == 'active'
        ).count()
        
        return {
            'active_bots': active_admins,
            'total_requests_last_hour': 15000,
            'average_response_time_ms': 234,
            'error_rate_percent': 0.5,
            'cache_hit_rate_percent': 78.5,
            'database_queries_per_second': 450,
            'timestamp': datetime_utils.now().isoformat()
        }
        
    finally:
        db.close()


@shared_task
def check_database_connections():
    """Check database connection pool status"""
    logger.info("Checking database connections...")
    
    try:
        db = SessionLocal()
        try:
            # Execute a simple query to check connection
            result = db.execute("SELECT count(*) FROM pg_stat_activity").scalar()
            
            # Get connection stats
            connections = db.execute(
                "SELECT state, count(*) FROM pg_stat_activity GROUP BY state"
            ).fetchall()
            
            connection_stats = {state: count for state, count in connections}
            
            return {
                'status': 'healthy',
                'total_connections': result,
                'active_connections': connection_stats.get('active', 0),
                'idle_connections': connection_stats.get('idle', 0),
                'idle_in_transaction': connection_stats.get('idle in transaction', 0),
                'max_connections': settings.DATABASE_POOL_SIZE,
                'checked_at': datetime_utils.now().isoformat()
            }
            
        finally:
            db.close()
            
    except Exception as e:
        logger.error(f"Database connection check failed: {e}")
        return {
            'status': 'unhealthy',
            'error': str(e),
            'checked_at': datetime_utils.now().isoformat()
        }


@shared_task
def check_redis_connection():
    """Check Redis connection status"""
    logger.info("Checking Redis connection...")
    
    try:
        import redis
        redis_client = redis.Redis(
            host=settings.REDIS_HOST,
            port=settings.REDIS_PORT,
            password=settings.REDIS_PASSWORD,
            socket_connect_timeout=5
        )
        
        # Test connection
        redis_client.ping()
        
        # Get info
        info = redis_client.info()
        
        return {
            'status': 'healthy',
            'version': info.get('redis_version'),
            'used_memory_mb': info.get('used_memory_human'),
            'connected_clients': info.get('connected_clients'),
            'uptime_days': info.get('uptime_in_days'),
            'hit_rate': redis_client.info('stats').get('keyspace_hitrate', 0),
            'checked_at': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Redis connection check failed: {e}")
        return {
            'status': 'unhealthy',
            'error': str(e),
            'checked_at': datetime_utils.now().isoformat()
        }