# backend/api/tasks/notification_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
import logging
import smtplib
from email.mime.text import MIMEText
from email.mime.multipart import MIMEMultipart

from core.database import SessionLocal
from core.models import Admin, Owner, NotificationLog, NotificationTemplate
from core.utils.datetime_utils import datetime_utils
from core.config import settings

logger = get_task_logger(__name__)


@shared_task
def send_telegram_notification(chat_id: int, message: str, parse_mode: str = 'HTML'):
    """Send Telegram notification"""
    logger.info(f"Sending Telegram notification to {chat_id}")
    
    # This would call Telegram Bot API
    # In production, this would use the bot instance
    
    # Mock sending
    success = True
    
    if success:
        logger.info(f"Telegram notification sent to {chat_id}")
        return {
            'channel': 'telegram',
            'recipient': chat_id,
            'status': 'sent',
            'sent_at': datetime_utils.now().isoformat()
        }
    else:
        logger.error(f"Failed to send Telegram notification to {chat_id}")
        return {
            'channel': 'telegram',
            'recipient': chat_id,
            'status': 'failed',
            'error': 'Failed to send'
        }


@shared_task
def send_email_notification(email: str, subject: str, body: str, html: bool = False):
    """Send email notification"""
    logger.info(f"Sending email notification to {email}")
    
    try:
        # Create message
        msg = MIMEMultipart('alternative')
        msg['Subject'] = subject
        msg['From'] = settings.EMAIL_FROM
        msg['To'] = email
        
        # Attach parts
        part = MIMEText(body, 'html' if html else 'plain')
        msg.attach(part)
        
        # Send email
        if settings.SMTP_HOST:
            server = smtplib.SMTP(settings.SMTP_HOST, settings.SMTP_PORT)
            if settings.SMTP_USE_TLS:
                server.starttls()
            if settings.SMTP_USER and settings.SMTP_PASSWORD:
                server.login(settings.SMTP_USER, settings.SMTP_PASSWORD)
            server.send_message(msg)
            server.quit()
        
        logger.info(f"Email notification sent to {email}")
        return {
            'channel': 'email',
            'recipient': email,
            'subject': subject,
            'status': 'sent',
            'sent_at': datetime_utils.now().isoformat()
        }
        
    except Exception as e:
        logger.error(f"Failed to send email to {email}: {e}")
        return {
            'channel': 'email',
            'recipient': email,
            'subject': subject,
            'status': 'failed',
            'error': str(e)
        }


@shared_task
def send_renewal_reminders():
    """Send renewal reminders to admins"""
    logger.info("Sending renewal reminders...")
    
    now = datetime_utils.now()
    reminder_days = [7, 3, 1]
    results = []
    
    db = SessionLocal()
    try:
        for days in reminder_days:
            expiry_date = now + timedelta(days=days)
            start_range = expiry_date - timedelta(days=1)
            end_range = expiry_date + timedelta(days=1)
            
            admins = db.query(Admin).filter(
                Admin.recharge_expiry.between(start_range, end_range),
                Admin.recharge_status == 'active'
            ).all()
            
            for admin in admins:
                # Create message based on days
                if days == 7:
                    subject = "Renewal Reminder - 7 Days Left"
                    message = f"Your admin panel will expire in 7 days. Please renew to avoid service interruption."
                elif days == 3:
                    subject = "URGENT: Renewal Required - 3 Days Left"
                    message = f"Your admin panel will expire in 3 days! Renew now to maintain service."
                elif days == 1:
                    subject = "FINAL WARNING: Renewal Tomorrow"
                    message = f"Your admin panel will expire TOMORROW! Immediate renewal required."
                else:
                    subject = f"Renewal Reminder - {days} Days Left"
                    message = f"Your admin panel will expire in {days} days."
                
                # Create notification log
                notification = NotificationLog(
                    admin_id=admin.id,
                    notification_type='renewal_reminder',
                    channel='telegram',
                    recipient_type='admin',
                    recipient_id=admin.telegram_id,
                    subject=subject,
                    message=message,
                    status='sent',
                    sent_at=datetime_utils.now()
                )
                db.add(notification)
                
                results.append({
                    'admin_id': admin.id,
                    'username': admin.admin_username,
                    'days_left': days,
                    'notification_id': notification.id
                })
                
                # Send actual notification
                if admin.telegram_id:
                    send_telegram_notification.delay(
                        admin.telegram_id,
                        f"🔔 **{subject}**\n\n{message}"
                    )
        
        db.commit()
        
        logger.info(f"Renewal reminders sent to {len(results)} admins")
        return {
            'sent_count': len(results),
            'details': results,
            'timestamp': datetime_utils.now().isoformat()
        }
        
    finally:
        db.close()


@shared_task
def send_low_key_alert(admin_id: int, reserve_id: str, duration: str, 
                       count: int, threshold: int):
    """Send low key alert to admin"""
    logger.info(f"Sending low key alert to admin {admin_id} for {duration} keys")
    
    message = (
        f"⚠️ **Low Key Alert**\n\n"
        f"**Bot:** {reserve_id}\n"
        f"**Duration:** {duration}\n"
        f"**Current Count:** {count}\n"
        f"**Threshold:** {threshold}\n\n"
        f"Please add more keys to avoid disruption."
    )
    
    db = SessionLocal()
    try:
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if admin and admin.telegram_id:
            # Send Telegram notification
            send_telegram_notification.delay(admin.telegram_id, message)
            
            # Log notification
            notification = NotificationLog(
                admin_id=admin_id,
                notification_type='low_key_alert',
                channel='telegram',
                recipient_type='admin',
                recipient_id=admin.telegram_id,
                subject=f'Low Key Alert: {duration}',
                message=message,
                status='sent',
                sent_at=datetime_utils.now()
            )
            db.add(notification)
            db.commit()
            
            logger.info(f"Low key alert sent to admin {admin_id}")
            
            return {
                'admin_id': admin_id,
                'reserve_id': reserve_id,
                'duration': duration,
                'notification_sent': True,
                'sent_at': datetime_utils.now().isoformat()
            }
        
        return {
            'admin_id': admin_id,
            'reserve_id': reserve_id,
            'duration': duration,
            'notification_sent': False,
            'reason': 'Admin not found or no telegram_id'
        }
        
    finally:
        db.close()


@shared_task
def send_payment_notification(admin_id: int, transaction_data: dict):
    """Send payment notification to admin"""
    logger.info(f"Sending payment notification to admin {admin_id}")
    
    amount = transaction_data.get('amount', 0)
    user_id = transaction_data.get('user_id', 'Unknown')
    
    message = (
        f"💰 **New Payment Received**\n\n"
        f"**Amount:** ₹{amount}\n"
        f"**User:** {user_id}\n"
        f"**Time:** {datetime_utils.now().strftime('%Y-%m-%d %H:%M:%S')}\n\n"
        f"Key has been delivered automatically."
    )
    
    db = SessionLocal()
    try:
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if admin and admin.telegram_id:
            send_telegram_notification.delay(admin.telegram_id, message)
            
            logger.info(f"Payment notification sent to admin {admin_id}")
            return {'sent': True, 'admin_id': admin_id}
        
        return {'sent': False, 'admin_id': admin_id, 'reason': 'No telegram_id'}
        
    finally:
        db.close()


@shared_task
def send_welcome_notification(admin_id: int, username: str):
    """Send welcome notification to new admin"""
    logger.info(f"Sending welcome notification to admin {admin_id}")
    
    message = (
        f"🎉 **Welcome to Telegram Bot System!**\n\n"
        f"Hello {username},\n\n"
        f"Your admin panel has been created successfully.\n"
        f"Login at: https://your-domain.com/admin\n\n"
        f"Please complete your setup:\n"
        f"1. Configure payment gateway\n"
        f"2. Upload keys\n"
        f"3. Set up your bot\n"
        f"4. Upload APK\n\n"
        f"Need help? Contact support."
    )
    
    db = SessionLocal()
    try:
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if admin and admin.telegram_id:
            send_telegram_notification.delay(admin.telegram_id, message)
            
            logger.info(f"Welcome notification sent to admin {admin_id}")
            return {'sent': True, 'admin_id': admin_id}
        
        return {'sent': False, 'admin_id': admin_id, 'reason': 'No telegram_id'}
        
    finally:
        db.close()


@shared_task
def send_test_notification(admin_id: int):
    """Send test notification to admin"""
    logger.info(f"Sending test notification to admin {admin_id}")
    
    message = (
        f"🧪 **Test Notification**\n\n"
        f"This is a test message from the notification system.\n"
        f"If you receive this, notifications are working correctly.\n\n"
        f"Time: {datetime_utils.now().strftime('%Y-%m-%d %H:%M:%S UTC')}"
    )
    
    db = SessionLocal()
    try:
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        
        if admin and admin.telegram_id:
            send_telegram_notification.delay(admin.telegram_id, message)
            
            logger.info(f"Test notification sent to admin {admin_id}")
            return {'sent': True, 'admin_id': admin_id}
        
        return {'sent': False, 'admin_id': admin_id, 'reason': 'No telegram_id'}
        
    finally:
        db.close()