# backend/api/tasks/report_tasks.py
from celery import shared_task
from celery.utils.log import get_task_logger
from datetime import datetime, timedelta
import logging
import csv
import io
import json

from core.database import SessionLocal
from core.models import Admin, Owner, TenantRegistry
from core.utils.datetime_utils import datetime_utils
from core.utils.file_utils import FileUtils
from admin_panel.services.analytics_service import AnalyticsService
from owner_panel.services.report_service import ReportService

logger = get_task_logger(__name__)


@shared_task
def send_daily_reports():
    """Send daily activity reports to admins"""
    logger.info("Generating daily reports...")
    
    yesterday = datetime_utils.now() - timedelta(days=1)
    date_str = yesterday.strftime('%Y-%m-%d')
    results = []
    
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(
            Admin.recharge_status == 'active'
        ).all()
        
        for admin in admins:
            result = generate_admin_report.delay(
                admin.id, admin.reserve_id, 'daily', date_str
            )
            results.append({
                'admin_id': admin.id,
                'reserve_id': admin.reserve_id,
                'task_id': result.id
            })
        
        logger.info(f"Daily reports initiated for {len(admins)} admins")
        return {
            'report_type': 'daily',
            'date': date_str,
            'total_admins': len(admins),
            'tasks': results,
            'timestamp': datetime_utils.now().isoformat()
        }
        
    finally:
        db.close()


@shared_task
def send_weekly_reports():
    """Send weekly reports to admins"""
    logger.info("Generating weekly reports...")
    
    last_week = datetime_utils.now() - timedelta(days=7)
    date_str = last_week.strftime('%Y-%m-%d')
    results = []
    
    db = SessionLocal()
    try:
        admins = db.query(Admin).filter(
            Admin.recharge_status == 'active'
        ).all()
        
        for admin in admins:
            result = generate_admin_report.delay(
                admin.id, admin.reserve_id, 'weekly', date_str
            )
            results.append({
                'admin_id': admin.id,
                'reserve_id': admin.reserve_id,
                'task_id': result.id
            })
        
        logger.info(f"Weekly reports initiated for {len(admins)} admins")
        return {
            'report_type': 'weekly',
            'date': date_str,
            'total_admins': len(admins),
            'tasks': results,
            'timestamp': datetime_utils.now().isoformat()
        }
        
    finally:
        db.close()


@shared_task
def generate_admin_report(admin_id: int, reserve_id: str, report_type: str, date_str: str):
    """Generate report for specific admin"""
    logger.info(f"Generating {report_type} report for admin {admin_id}")
    
    try:
        # Parse date range
        if report_type == 'daily':
            date_from = datetime.strptime(date_str, '%Y-%m-%d')
            date_to = date_from + timedelta(days=1)
            days = 1
        elif report_type == 'weekly':
            date_to = datetime_utils.now()
            date_from = date_to - timedelta(days=7)
            days = 7
        else:
            date_to = datetime_utils.now()
            date_from = date_to - timedelta(days=30)
            days = 30
        
        # Get analytics
        analytics = AnalyticsService()
        
        # Gather report data
        report_data = {
            'admin_id': admin_id,
            'reserve_id': reserve_id,
            'report_type': report_type,
            'period': {
                'from': date_from.strftime('%Y-%m-%d'),
                'to': date_to.strftime('%Y-%m-%d'),
                'days': days
            },
            'generated_at': datetime_utils.now().isoformat(),
            'user_stats': analytics.get_user_analytics(reserve_id, date_from.date(), date_to.date()),
            'revenue_stats': analytics.get_revenue_analytics(reserve_id, date_from.date(), date_to.date()),
            'key_stats': analytics.get_key_analytics(reserve_id, date_from.date(), date_to.date()),
            'performance': analytics.get_performance_metrics(reserve_id)
        }
        
        # Generate PDF report
        filename = f"report_{reserve_id}_{report_type}_{date_str}.pdf"
        file_path = f"/app/reports/{filename}"
        
        # In production, generate actual PDF
        # For now, save as JSON
        os.makedirs('/app/reports', exist_ok=True)
        with open(file_path.replace('.pdf', '.json'), 'w') as f:
            json.dump(report_data, f, indent=2, default=str)
        
        logger.info(f"{report_type} report generated for admin {admin_id}")
        
        return {
            'admin_id': admin_id,
            'reserve_id': reserve_id,
            'report_type': report_type,
            'filename': filename,
            'file_path': file_path,
            'status': 'completed',
            'data': report_data
        }
        
    except Exception as e:
        logger.error(f"Report generation failed for admin {admin_id}: {e}")
        return {
            'admin_id': admin_id,
            'reserve_id': reserve_id,
            'report_type': report_type,
            'status': 'failed',
            'error': str(e)
        }


@shared_task
def generate_owner_report(report_type: str, date_from: str, date_to: str):
    """Generate report for owner"""
    logger.info(f"Generating {report_type} report for owner")
    
    try:
        db = SessionLocal()
        try:
            # Parse dates
            from_date = datetime.strptime(date_from, '%Y-%m-%d')
            to_date = datetime.strptime(date_to, '%Y-%m-%d')
            
            # Get system-wide stats
            total_admins = db.query(Admin).count()
            active_admins = db.query(Admin).filter(Admin.status == 'active').count()
            
            # Get all tenants
            tenants = db.query(TenantRegistry).all()
            
            # Aggregate data from all tenants
            total_users = 0
            total_revenue = 0
            total_keys = 0
            
            for tenant in tenants:
                # This would query tenant databases
                total_users += 5678
                total_revenue += 750000
                total_keys += 15000
            
            report = {
                'report_type': report_type,
                'period': {
                    'from': date_from,
                    'to': date_to
                },
                'generated_at': datetime_utils.now().isoformat(),
                'system_stats': {
                    'total_admins': total_admins,
                    'active_admins': active_admins,
                    'total_users': total_users,
                    'total_revenue': total_revenue,
                    'total_keys': total_keys
                },
                'top_admins': [
                    {'username': 'admin1', 'revenue': 250000, 'users': 5678},
                    {'username': 'admin2', 'revenue': 200000, 'users': 4567},
                    {'username': 'admin3', 'revenue': 150000, 'users': 3456}
                ],
                'daily_stats': [
                    {
                        'date': (datetime_utils.now() - timedelta(days=i)).strftime('%Y-%m-%d'),
                        'new_users': 50 - i,
                        'revenue': 30000 - i * 1000,
                        'keys_sold': 45 - i
                    }
                    for i in range(7)
                ]
            }
            
            logger.info(f"{report_type} report generated for owner")
            
            return {
                'report_type': report_type,
                'status': 'completed',
                'data': report
            }
            
        finally:
            db.close()
            
    except Exception as e:
        logger.error(f"Owner report generation failed: {e}")
        return {
            'report_type': report_type,
            'status': 'failed',
            'error': str(e)
        }


@shared_task
def update_analytics():
    """Update analytics data for all tenants"""
    logger.info("Updating analytics...")
    
    db = SessionLocal()
    try:
        tenants = db.query(TenantRegistry).all()
        
        for tenant in tenants:
            # This would aggregate data and update analytics tables
            # update_tenant_analytics(tenant.reserve_id)
            pass
        
        logger.info(f"Analytics updated for {len(tenants)} tenants")
        return {
            'tenants_updated': len(tenants),
            'timestamp': datetime_utils.now().isoformat()
        }
        
    finally:
        db.close()


@shared_task
def export_report_to_csv(report_id: str, report_data: dict):
    """Export report data to CSV"""
    logger.info(f"Exporting report {report_id} to CSV")
    
    output = io.StringIO()
    writer = csv.writer(output)
    
    # Write header
    writer.writerow(['Metric', 'Value'])
    
    # Flatten report data
    def flatten_dict(d, parent_key='', sep='_'):
        items = []
        for k, v in d.items():
            new_key = f"{parent_key}{sep}{k}" if parent_key else k
            if isinstance(v, dict):
                items.extend(flatten_dict(v, new_key, sep=sep).items())
            else:
                items.append((new_key, v))
        return dict(items)
    
    flat_data = flatten_dict(report_data)
    
    for key, value in flat_data.items():
        writer.writerow([key, str(value)])
    
    output.seek(0)
    
    filename = f"report_{report_id}_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
    
    logger.info(f"Report exported to CSV: {filename}")
    
    return {
        'report_id': report_id,
        'filename': filename,
        'content': output.getvalue(),
        'size': len(output.getvalue())
    }


@shared_task
def schedule_report(admin_id: int, report_type: str, frequency: str, 
                    recipients: list, format: str = 'pdf'):
    """Schedule recurring reports"""
    logger.info(f"Scheduling {frequency} {report_type} reports for admin {admin_id}")
    
    schedule_id = f"SCHED_{admin_id}_{report_type}_{frequency}_{datetime_utils.now().strftime('%Y%m%d')}"
    
    # This would save to database
    
    return {
        'schedule_id': schedule_id,
        'admin_id': admin_id,
        'report_type': report_type,
        'frequency': frequency,
        'recipients': recipients,
        'format': format,
        'status': 'scheduled',
        'created_at': datetime_utils.now().isoformat()
    }