# backend/api/v1/activity.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional
from datetime import date

from core.database import get_db
from core.security import get_current_owner, get_current_admin
from owner_panel.controllers import ActivityController as OwnerActivityController
from admin_panel.controllers import UserController as AdminUserController

router = APIRouter(prefix="/activity", tags=["Activity"])

# Owner endpoints
@router.get("/owner/all")
async def get_all_activities(
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    activity_type: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    activity_controller: OwnerActivityController = Depends()
):
    """Get all activities (owner only)"""
    filters = {
        'date_from': date_from,
        'date_to': date_to,
        'activity_type': activity_type
    }
    return await activity_controller.get_all_activities(filters, skip, limit)

@router.get("/owner/admin")
async def get_admin_activities(
    admin_id: Optional[int] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    activity_type: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    activity_controller: OwnerActivityController = Depends()
):
    """Get admin activities (owner only)"""
    return await activity_controller.get_admin_activities(
        admin_id, date_from, date_to, activity_type, skip, limit
    )

@router.get("/owner/user")
async def get_user_activities(
    user_id: Optional[int] = None,
    bot: Optional[str] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    activity_type: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    activity_controller: OwnerActivityController = Depends()
):
    """Get user activities (owner only)"""
    return await activity_controller.get_user_activities(
        user_id, bot, date_from, date_to, activity_type, skip, limit
    )

# Admin endpoints
@router.get("/admin")
async def get_my_user_activities(
    user_id: Optional[int] = None,
    activity_type: Optional[str] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    user_controller: AdminUserController = Depends()
):
    """Get user activities for current admin's bot"""
    return await user_controller.get_activity(
        current_admin.reserve_id, user_id, activity_type,
        date_from, date_to, skip, limit
    )

@router.get("/admin/stats")
async def get_activity_stats(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    user_controller: AdminUserController = Depends()
):
    """Get activity statistics for current admin's bot"""
    return await user_controller.get_activity_stats(
        current_admin.reserve_id, date_from, date_to
    )