# backend/api/v1/admin.py
from fastapi import APIRouter, Depends, HTTPException, Query, UploadFile, File
from typing import List, Optional

from core.database import get_db
from core.security import get_current_admin
from admin_panel.controllers import (
    UserController, KeyController, PaymentController,
    APKController, BroadcastController, AnalyticsController
)
from admin_panel.schemas import (
    UserResponse, KeyCreate, KeyResponse,
    TransactionResponse, BroadcastCreate
)

router = APIRouter(prefix="/admin", tags=["Admin"])

@router.get("/dashboard")
async def get_dashboard(
    current_admin = Depends(get_current_admin),
    analytics_controller: AnalyticsController = Depends()
):
    """Get admin dashboard data"""
    return await analytics_controller.get_dashboard(current_admin.reserve_id)

@router.get("/users")
async def get_users(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    search: Optional[str] = None,
    current_admin = Depends(get_current_admin),
    user_controller: UserController = Depends()
):
    """Get all users"""
    return await user_controller.get_users(
        current_admin.reserve_id, skip, limit, search, None
    )

@router.post("/keys/upload")
async def upload_keys(
    file: UploadFile = File(...),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Upload stored keys"""
    return await key_controller.upload_stored_keys(
        current_admin.reserve_id, file
    )

@router.post("/keys/generate")
async def generate_keys(
    key_data: KeyCreate,
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Generate new keys"""
    return await key_controller.generate_keys(
        current_admin.reserve_id, key_data
    )

@router.get("/payments")
async def get_payments(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    payment_controller: PaymentController = Depends()
):
    """Get payment transactions"""
    return await payment_controller.get_transactions(
        current_admin.reserve_id, skip, limit
    )

@router.post("/broadcast")
async def create_broadcast(
    broadcast_data: BroadcastCreate,
    current_admin = Depends(get_current_admin),
    broadcast_controller: BroadcastController = Depends()
):
    """Create and send broadcast"""
    return await broadcast_controller.create_broadcast(
        current_admin.reserve_id, current_admin.id, broadcast_data
    )

@router.post("/apk/upload")
async def upload_apk(
    file: UploadFile = File(...),
    version_name: str = Form(...),
    version_code: int = Form(...),
    release_notes: Optional[str] = Form(None),
    current_admin = Depends(get_current_admin),
    apk_controller: APKController = Depends()
):
    """Upload APK file"""
    return await apk_controller.upload_apk(
        current_admin.reserve_id, file, version_name,
        version_code, release_notes
    )