# backend/api/v1/broadcasts.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional
from datetime import datetime

from core.database import get_db
from core.security import get_current_admin
from admin_panel.controllers import BroadcastController
from admin_panel.schemas import BroadcastCreate

router = APIRouter(prefix="/broadcasts", tags=["Broadcasts"])

@router.post("/")
async def create_broadcast(
    broadcast_data: BroadcastCreate,
    current_admin = Depends(get_current_admin),
    broadcast_controller: BroadcastController = Depends()
):
    """Create and send broadcast"""
    return await broadcast_controller.create_broadcast(
        current_admin.reserve_id, current_admin.id, broadcast_data.dict()
    )

@router.get("/history")
async def get_broadcast_history(
    skip: int = Query(0, ge=0),
    limit: int = Query(50, ge=1, le=200),
    current_admin = Depends(get_current_admin),
    broadcast_controller: BroadcastController = Depends()
):
    """Get broadcast history"""
    return await broadcast_controller.get_history(
        current_admin.reserve_id, skip, limit
    )

@router.get("/{broadcast_id}/stats")
async def get_broadcast_stats(
    broadcast_id: str,
    current_admin = Depends(get_current_admin),
    broadcast_controller: BroadcastController = Depends()
):
    """Get broadcast statistics"""
    return await broadcast_controller.get_stats(
        current_admin.reserve_id, broadcast_id
    )

@router.post("/{broadcast_id}/cancel")
async def cancel_broadcast(
    broadcast_id: str,
    current_admin = Depends(get_current_admin),
    broadcast_controller: BroadcastController = Depends()
):
    """Cancel scheduled broadcast"""
    return await broadcast_controller.cancel_broadcast(
        current_admin.reserve_id, broadcast_id
    )