# backend/api/v1/keys.py
from fastapi import APIRouter, Depends, HTTPException, Query, UploadFile, File
from typing import List, Optional

from core.database import get_db
from core.security import get_current_admin
from admin_panel.controllers import KeyController
from admin_panel.schemas import KeyCreate, KeyResponse

router = APIRouter(prefix="/keys", tags=["Keys"])

@router.post("/stored/upload")
async def upload_stored_keys(
    file: UploadFile = File(...),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Upload CSV file with stored keys"""
    return await key_controller.upload_stored_keys(
        current_admin.reserve_id, file
    )

@router.get("/stored")
async def get_stored_keys(
    duration: Optional[str] = None,
    status: Optional[str] = None,
    batch_id: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Get stored keys inventory"""
    return await key_controller.get_stored_keys(
        current_admin.reserve_id, duration, status, batch_id, skip, limit
    )

@router.post("/generated")
async def generate_keys(
    key_data: KeyCreate,
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Generate new keys"""
    return await key_controller.generate_keys(
        current_admin.reserve_id, key_data.dict()
    )

@router.get("/generated/pattern")
async def get_key_pattern(
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Get current key generation pattern"""
    return await key_controller.get_key_pattern(current_admin.reserve_id)

@router.post("/generated/pattern")
async def set_key_pattern(
    pattern: str = Query(...),
    prefix: Optional[str] = None,
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Set key generation pattern"""
    pattern_data = {
        'pattern': pattern,
        'prefix': prefix,
        'exclude_similar': True,
        'checksum_enabled': False
    }
    return await key_controller.set_key_pattern(
        current_admin.reserve_id, pattern_data
    )

@router.get("/")
async def get_all_keys(
    key_type: Optional[str] = None,
    duration: Optional[str] = None,
    status: Optional[str] = None,
    user_id: Optional[int] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Get all keys with filters"""
    return await key_controller.get_all_keys(
        current_admin.reserve_id, key_type, duration, status, 
        user_id, skip, limit
    )

@router.post("/{key_id}/block")
async def block_key(
    key_id: int,
    reason: str = Query(...),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Block a key"""
    return await key_controller.block_key(
        current_admin.reserve_id, key_id, reason
    )

@router.post("/{key_id}/unblock")
async def unblock_key(
    key_id: int,
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Unblock a key"""
    return await key_controller.unblock_key(
        current_admin.reserve_id, key_id
    )

@router.post("/{key_id}/extend")
async def extend_key(
    key_id: int,
    days: int = Query(..., ge=1, le=365),
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Extend key validity"""
    return await key_controller.extend_key(
        current_admin.reserve_id, key_id, days
    )

@router.get("/stats")
async def get_key_stats(
    current_admin = Depends(get_current_admin),
    key_controller: KeyController = Depends()
):
    """Get key statistics"""
    return await key_controller.get_key_stats(current_admin.reserve_id)