# backend/api/v1/monitoring.py
from fastapi import APIRouter, Depends, HTTPException, Query, Request
from fastapi.responses import Response
from typing import Optional, Dict, Any
import psutil
import platform
import time
from datetime import datetime, timedelta
import os

from core.database import get_db, engine
from core.security import get_current_owner
from prometheus_client import generate_latest, REGISTRY, Counter, Histogram, Gauge

router = APIRouter(prefix="/monitoring", tags=["Monitoring"])

# Prometheus metrics
http_requests_total = Counter('http_requests_total', 'Total HTTP requests', ['method', 'endpoint', 'status'])
http_request_duration_seconds = Histogram('http_request_duration_seconds', 'HTTP request duration', ['method', 'endpoint'])
active_users = Gauge('active_users', 'Number of active users')
system_cpu_usage = Gauge('system_cpu_usage', 'CPU usage percentage')
system_memory_usage = Gauge('system_memory_usage', 'Memory usage percentage')
system_disk_usage = Gauge('system_disk_usage', 'Disk usage percentage')


@router.get("/health")
async def health_check():
    """Health check endpoint"""
    # Check database
    db_status = "up"
    try:
        with engine.connect() as conn:
            conn.execute("SELECT 1")
    except:
        db_status = "down"
    
    # Check disk space
    disk = psutil.disk_usage('/')
    disk_status = "up" if disk.percent < 90 else "degraded"
    
    # Check memory
    memory = psutil.virtual_memory()
    memory_status = "up" if memory.percent < 85 else "degraded"
    
    status = "healthy"
    if db_status == "down" or disk_status == "down" or memory_status == "down":
        status = "unhealthy"
    elif disk_status == "degraded" or memory_status == "degraded":
        status = "degraded"
    
    return {
        "status": status,
        "timestamp": datetime.utcnow().isoformat(),
        "checks": {
            "database": db_status,
            "disk": disk_status,
            "memory": memory_status,
            "cpu": "up" if psutil.cpu_percent() < 80 else "degraded"
        },
        "version": os.getenv("APP_VERSION", "1.0.0"),
        "environment": os.getenv("ENVIRONMENT", "development")
    }


@router.get("/metrics")
async def get_metrics():
    """Prometheus metrics endpoint"""
    # Update system metrics
    system_cpu_usage.set(psutil.cpu_percent())
    system_memory_usage.set(psutil.virtual_memory().percent)
    system_disk_usage.set(psutil.disk_usage('/').percent)
    
    return Response(content=generate_latest(REGISTRY), media_type="text/plain")


@router.get("/system")
async def get_system_info(
    current_owner = Depends(get_current_owner)
):
    """Get detailed system information"""
    boot_time = datetime.fromtimestamp(psutil.boot_time())
    uptime = datetime.utcnow() - boot_time
    
    cpu_freq = psutil.cpu_freq()
    memory = psutil.virtual_memory()
    disk = psutil.disk_usage('/')
    network = psutil.net_io_counters()
    
    return {
        "system": {
            "hostname": platform.node(),
            "platform": platform.system(),
            "release": platform.release(),
            "version": platform.version(),
            "architecture": platform.machine(),
            "processor": platform.processor(),
            "boot_time": boot_time.isoformat(),
            "uptime_seconds": uptime.total_seconds(),
            "uptime_human": str(uptime).split('.')[0]
        },
        "cpu": {
            "count": psutil.cpu_count(),
            "physical_count": psutil.cpu_count(logical=False),
            "percent": psutil.cpu_percent(interval=1),
            "percent_per_cpu": psutil.cpu_percent(interval=1, percpu=True),
            "frequency": {
                "current": cpu_freq.current if cpu_freq else 0,
                "min": cpu_freq.min if cpu_freq else 0,
                "max": cpu_freq.max if cpu_freq else 0
            } if cpu_freq else None,
            "stats": psutil.cpu_stats()._asdict()
        },
        "memory": {
            "total": memory.total,
            "available": memory.available,
            "used": memory.used,
            "percent": memory.percent,
            "swap": psutil.swap_memory()._asdict()
        },
        "disk": {
            "total": disk.total,
            "used": disk.used,
            "free": disk.free,
            "percent": disk.percent,
            "partitions": [
                {
                    "device": p.device,
                    "mountpoint": p.mountpoint,
                    "fstype": p.fstype,
                    "opts": p.opts
                }
                for p in psutil.disk_partitions()
            ]
        },
        "network": {
            "bytes_sent": network.bytes_sent,
            "bytes_recv": network.bytes_recv,
            "packets_sent": network.packets_sent,
            "packets_recv": network.packets_recv,
            "connections": len(psutil.net_connections()),
            "interfaces": psutil.net_if_stats()
        },
        "processes": len(psutil.pids())
    }


@router.get("/performance")
async def get_performance_metrics(
    minutes: int = Query(5, ge=1, le=60),
    current_owner = Depends(get_current_owner)
):
    """Get performance metrics for last N minutes"""
    # This would query a time-series database in production
    # For now, return mock data
    now = datetime.utcnow()
    data = []
    
    for i in range(minutes):
        timestamp = now - timedelta(minutes=minutes - i - 1)
        data.append({
            "timestamp": timestamp.isoformat(),
            "cpu": 30 + (i % 20),
            "memory": 40 + (i % 15),
            "requests": 100 + (i * 10),
            "response_time": 200 + (i * 5)
        })
    
    return {
        "period_minutes": minutes,
        "metrics": data
    }


@router.get("/alerts")
async def get_active_alerts(
    severity: Optional[str] = None,
    current_owner = Depends(get_current_owner)
):
    """Get active alerts"""
    # This would query alert manager
    alerts = [
        {
            "id": 1,
            "name": "High CPU Usage",
            "severity": "warning",
            "status": "firing",
            "starts_at": (datetime.utcnow() - timedelta(minutes=5)).isoformat(),
            "value": "85%",
            "instance": "backend-01"
        },
        {
            "id": 2,
            "name": "Low Disk Space",
            "severity": "critical",
            "status": "firing",
            "starts_at": (datetime.utcnow() - timedelta(hours=2)).isoformat(),
            "value": "92%",
            "instance": "postgres-01"
        }
    ]
    
    if severity:
        alerts = [a for a in alerts if a["severity"] == severity]
    
    return {"alerts": alerts, "total": len(alerts)}


@router.get("/logs")
async def get_recent_logs(
    level: Optional[str] = Query(None, regex="^(DEBUG|INFO|WARNING|ERROR)$"),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner)
):
    """Get recent application logs"""
    # This would query Elasticsearch in production
    # For now, return mock data
    logs = []
    
    for i in range(min(limit, 100)):
        log_levels = ["INFO", "DEBUG", "WARNING", "ERROR"]
        level_choice = level if level else log_levels[i % 4]
        
        logs.append({
            "timestamp": (datetime.utcnow() - timedelta(seconds=i*10)).isoformat(),
            "level": level_choice,
            "logger": "backend.api",
            "message": f"Sample log message {i}",
            "module": "api.v1.monitoring",
            "line": 123
        })
    
    return {"logs": logs, "total": len(logs)}


@router.post("/alert-rules")
async def create_alert_rule(
    rule_data: Dict[str, Any],
    current_owner = Depends(get_current_owner)
):
    """Create new alert rule"""
    # This would save to database
    return {
        "id": 1,
        "name": rule_data.get("name"),
        "metric": rule_data.get("metric"),
        "threshold": rule_data.get("threshold"),
        "severity": rule_data.get("severity"),
        "created_at": datetime.utcnow().isoformat()
    }


@router.delete("/alert-rules/{rule_id}")
async def delete_alert_rule(
    rule_id: int,
    current_owner = Depends(get_current_owner)
):
    """Delete alert rule"""
    return {"message": f"Alert rule {rule_id} deleted"}