# backend/api/v1/owner.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional

from core.database import get_db
from core.security import get_current_owner
from owner_panel.controllers import (
    AdminController, UserController, ActivityController,
    RechargeController, SystemController
)
from owner_panel.schemas import (
    AdminCreate, AdminResponse, AdminUpdate,
    UserResponse, ActivityLogResponse,
    RechargePlanResponse, RechargeRequest
)

router = APIRouter(prefix="/owner", tags=["Owner"])

@router.get("/dashboard")
async def get_dashboard(
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Get owner dashboard data"""
    return await system_controller.get_dashboard_stats()

@router.post("/admins", response_model=AdminResponse)
async def create_admin(
    admin_data: AdminCreate,
    current_owner = Depends(get_current_owner),
    admin_controller: AdminController = Depends()
):
    """Create new admin"""
    return await admin_controller.create_admin(admin_data, current_owner.id)

@router.get("/admins", response_model=List[AdminResponse])
async def get_admins(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    status: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    admin_controller: AdminController = Depends()
):
    """Get all admins"""
    return await admin_controller.get_admins(skip, limit, status)

@router.get("/users")
async def get_users(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    search: Optional[str] = None,
    bot: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    user_controller: UserController = Depends()
):
    """Get all users"""
    return await user_controller.get_all_users(skip, limit, search, bot)

@router.get("/activities")
async def get_activities(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    date_from: Optional[str] = None,
    date_to: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    activity_controller: ActivityController = Depends()
):
    """Get all activities"""
    filters = {
        'date_from': date_from,
        'date_to': date_to
    }
    return await activity_controller.get_all_activities(filters, skip, limit)

@router.post("/recharge/{admin_id}")
async def recharge_admin(
    admin_id: int,
    recharge_data: RechargeRequest,
    current_owner = Depends(get_current_owner),
    recharge_controller: RechargeController = Depends()
):
    """Recharge admin panel"""
    return await recharge_controller.recharge_admin(
        admin_id, recharge_data, current_owner.id
    )

@router.get("/settings")
async def get_settings(
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Get system settings"""
    return await system_controller.get_settings()