# backend/api/v1/payments.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional
from datetime import date

from core.database import get_db
from core.security import get_current_admin, get_current_user
from admin_panel.controllers import PaymentController

router = APIRouter(prefix="/payments", tags=["Payments"])

@router.get("/")
async def get_transactions(
    status: Optional[str] = None,
    user_id: Optional[int] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_admin = Depends(get_current_admin),
    payment_controller: PaymentController = Depends()
):
    """Get all transactions"""
    return await payment_controller.get_transactions(
        current_admin.reserve_id, status, user_id, 
        date_from, date_to, skip, limit
    )

@router.get("/{transaction_id}")
async def get_transaction(
    transaction_id: str,
    current_admin = Depends(get_current_admin),
    payment_controller: PaymentController = Depends()
):
    """Get transaction details"""
    return await payment_controller.get_transaction(
        current_admin.reserve_id, transaction_id
    )

@router.post("/{transaction_id}/refund")
async def refund_transaction(
    transaction_id: str,
    amount: Optional[float] = None,
    reason: str = Query(...),
    current_admin = Depends(get_current_admin),
    payment_controller: PaymentController = Depends()
):
    """Process refund"""
    return await payment_controller.process_refund(
        current_admin.reserve_id, transaction_id, amount, reason
    )

@router.get("/stats")
async def get_payment_stats(
    date_from: date,
    date_to: date,
    current_admin = Depends(get_current_admin),
    payment_controller: PaymentController = Depends()
):
    """Get payment statistics"""
    return await payment_controller.get_payment_stats(
        current_admin.reserve_id, date_from, date_to
    )

@router.post("/webhook/{gateway}")
async def payment_webhook(
    gateway: str,
    request: Request
):
    """Handle payment gateway webhook"""
    # Verify webhook signature
    # Update payment status
    return {'status': 'received'}