# backend/api/v1/system.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional

from core.database import get_db
from core.security import get_current_owner
from owner_panel.controllers import SystemController
from owner_panel.schemas import SystemSettingsUpdate

router = APIRouter(prefix="/system", tags=["System"])

@router.get("/settings")
async def get_settings(
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Get system settings"""
    return await system_controller.get_settings()

@router.put("/settings")
async def update_settings(
    settings_update: SystemSettingsUpdate,
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Update system settings"""
    return await system_controller.update_settings(
        settings_update.settings, current_owner.id
    )

@router.post("/backup")
async def create_backup(
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Create system backup"""
    return await system_controller.create_backup(current_owner.id)

@router.get("/backups")
async def get_backups(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db)
):
    """Get backup history"""
    from core.models import BackupHistory
    
    backups = db.query(BackupHistory).order_by(
        BackupHistory.created_at.desc()
    ).offset(skip).limit(limit).all()
    
    return [
        {
            'id': b.id,
            'backup_id': b.backup_id,
            'type': b.backup_type,
            'size': b.backup_size,
            'status': b.status,
            'created_at': b.created_at.isoformat(),
            'completed_at': b.completed_at.isoformat() if b.completed_at else None
        }
        for b in backups
    ]

@router.get("/audit-logs")
async def get_audit_logs(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Get audit logs"""
    return await system_controller.get_audit_logs(skip, limit)

@router.post("/maintenance")
async def set_maintenance_mode(
    enabled: bool = Query(...),
    current_owner = Depends(get_current_owner),
    system_controller: SystemController = Depends()
):
    """Set maintenance mode"""
    await system_controller.update_settings(
        {'maintenance_mode': str(enabled).lower()},
        current_owner.id
    )
    return {'maintenance_mode': enabled}