# backend/api/v1/user.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional

from core.database import get_db
from core.security import get_current_user
from bot_core.services.user_service import UserService
from bot_core.services.payment_processor import PaymentProcessor
from bot_core.services.key_delivery import KeyDeliveryService

router = APIRouter(prefix="/user", tags=["User"])

@router.get("/plans")
async def get_plans(
    current_user = Depends(get_current_user)
):
    """Get available plans"""
    from shared.constants import DEFAULT_PRICES
    return {
        'plans': [
            {'duration': '1 Day', 'price': DEFAULT_PRICES['1d'], 'description': 'Quick access'},
            {'duration': '3 Days', 'price': DEFAULT_PRICES['3d'], 'description': 'Weekend gaming'},
            {'duration': '7 Days', 'price': DEFAULT_PRICES['7d'], 'description': 'Weekly pass'},
            {'duration': '30 Days', 'price': DEFAULT_PRICES['30d'], 'description': 'Monthly saving'},
            {'duration': '60 Days', 'price': DEFAULT_PRICES['60d'], 'description': 'Best value'}
        ]
    }

@router.post("/purchase")
async def purchase_key(
    plan: str = Query(...),
    current_user = Depends(get_current_user),
    payment_processor: PaymentProcessor = Depends()
):
    """Purchase a key"""
    # This would create payment order
    payment_data = await payment_processor.create_payment(
        user_id=current_user.id,
        amount=600,  # Get from plan
        currency='INR',
        description=f"Purchase {plan} Key"
    )
    return payment_data

@router.get("/keys")
async def get_my_keys(
    current_user = Depends(get_current_user),
    key_service: KeyDeliveryService = Depends()
):
    """Get user's keys"""
    # This would get user's keys from database
    return {'keys': []}

@router.get("/download")
async def get_download_link(
    current_user = Depends(get_current_user),
    key_service: KeyDeliveryService = Depends()
):
    """Get APK download link"""
    # Check if user has active key
    has_active_key = True  # Check in database
    
    if not has_active_key:
        raise HTTPException(status_code=403, detail="Active key required for download")
    
    # Generate temporary download link
    download_token = "temp_token"  # Generate token
    download_url = f"https://your-domain.com/download/{download_token}"
    
    return {'download_url': download_url, 'expires_in': 24}

@router.post("/extend/{key_id}")
async def extend_key(
    key_id: int,
    days: int = Query(7, ge=1, le=60),
    current_user = Depends(get_current_user),
    key_service: KeyDeliveryService = Depends()
):
    """Extend key validity"""
    # This would process extension payment
    return {'key_id': key_id, 'days': days, 'status': 'pending'}

@router.post("/redeem")
async def redeem_key(
    key: str = Query(...),
    current_user = Depends(get_current_user),
    key_service: KeyDeliveryService = Depends()
):
    """Redeem a key"""
    # Validate and redeem key
    return {'key': key, 'status': 'redeemed'}

@router.post("/contact")
async def contact_support(
    message: str = Query(..., min_length=1, max_length=1000),
    current_user = Depends(get_current_user)
):
    """Contact support"""
    # Create support ticket
    return {'status': 'received', 'message': message}