# backend/bot_core/bot.py
import asyncio
import logging
from typing import Optional, Dict, Any
from telegram import Update, Bot, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import (
    Application, ApplicationBuilder, CommandHandler, 
    MessageHandler, CallbackQueryHandler, filters,
    ContextTypes, ConversationHandler
)
from telegram.constants import ParseMode

from config.settings import settings
from core.database import tenant_db_manager
from shared.constants.messages import BotMessages
from .handlers import (
    StartHandler, PlansHandler, BuyHandler, 
    KeysHandler, DownloadHandler, ExtendHandler,
    HelpHandler
)
from .services import ActivityTracker, UserService

logger = logging.getLogger(__name__)

# Conversation states
SELECTING_PLAN, CONFIRMING_PURCHASE, WAITING_PAYMENT, EXTENDING_KEY = range(4)

class TelegramBot:
    """Main Telegram bot class"""
    
    def __init__(self, admin_id: int, bot_token: str, reserve_id: str):
        self.admin_id = admin_id
        self.bot_token = bot_token
        self.reserve_id = reserve_id
        self.application: Optional[Application] = None
        self.db_url = None
        
        # Initialize handlers
        self.start_handler = StartHandler()
        self.plans_handler = PlansHandler()
        self.buy_handler = BuyHandler()
        self.keys_handler = KeysHandler()
        self.download_handler = DownloadHandler()
        self.extend_handler = ExtendHandler()
        self.help_handler = HelpHandler()
        
        # Initialize services
        self.activity_tracker = ActivityTracker(reserve_id)
        self.user_service = UserService(reserve_id)
    
    async def initialize(self, db_url: str):
        """Initialize bot with database connection"""
        self.db_url = db_url
        await self.user_service.initialize(db_url)
        await self.activity_tracker.initialize(db_url)
        
        # Build application
        self.application = (
            ApplicationBuilder()
            .token(self.bot_token)
            .concurrent_updates(True)
            .connection_pool_size(settings.TELEGRAM_MAX_CONNECTIONS)
            .pool_timeout(30)
            .build()
        )
        
        # Register handlers
        self._register_handlers()
        
        logger.info(f"Bot initialized for admin {self.admin_id} (Reserve: {self.reserve_id})")
    
    def _register_handlers(self):
        """Register all bot handlers"""
        
        # Command handlers
        self.application.add_handler(CommandHandler("start", self._start_command))
        self.application.add_handler(CommandHandler("plans", self._plans_command))
        self.application.add_handler(CommandHandler("buy", self._buy_command))
        self.application.add_handler(CommandHandler("my_keys", self._my_keys_command))
        self.application.add_handler(CommandHandler("download", self._download_command))
        self.application.add_handler(CommandHandler("extend", self._extend_command))
        self.application.add_handler(CommandHandler("help", self._help_command))
        self.application.add_handler(CommandHandler("contact", self._contact_command))
        
        # Conversation handler for purchase flow
        purchase_conv = ConversationHandler(
            entry_points=[CommandHandler("buy", self._buy_command)],
            states={
                SELECTING_PLAN: [CallbackQueryHandler(self._plan_selected, pattern="^plan_")],
                CONFIRMING_PURCHASE: [CallbackQueryHandler(self._confirm_purchase, pattern="^confirm_")],
                WAITING_PAYMENT: [MessageHandler(filters.TEXT & ~filters.COMMAND, self._check_payment)],
            },
            fallbacks=[CommandHandler("cancel", self._cancel)],
        )
        self.application.add_handler(purchase_conv)
        
        # Conversation handler for key extension
        extend_conv = ConversationHandler(
            entry_points=[CommandHandler("extend", self._extend_command)],
            states={
                EXTENDING_KEY: [CallbackQueryHandler(self._extend_key_selected, pattern="^extend_")],
            },
            fallbacks=[CommandHandler("cancel", self._cancel)],
        )
        self.application.add_handler(extend_conv)
        
        # Callback query handlers
        self.application.add_handler(CallbackQueryHandler(self._button_callback, pattern="^btn_"))
        
        # Message handlers
        self.application.add_handler(MessageHandler(filters.TEXT & ~filters.COMMAND, self._handle_message))
        self.application.add_handler(MessageHandler(filters.PHOTO, self._handle_photo))
        
        # Error handler
        self.application.add_error_handler(self._error_handler)
    
    async def _start_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /start command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(
            user.id, "start", {"username": user.username}
        )
        
        # Check if user is blocked
        if await self.user_service.is_blocked(user.id):
            await update.message.reply_text(BotMessages.USER_BLOCKED)
            return
        
        await self.start_handler.handle(update, context, self.reserve_id)
    
    async def _plans_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /plans command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "view_plans", {})
        
        await self.plans_handler.handle(update, context, self.reserve_id)
    
    async def _buy_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /buy command - start purchase flow"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "initiate_purchase", {})
        
        return await self.buy_handler.start(update, context, self.reserve_id)
    
    async def _my_keys_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /my_keys command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "view_keys", {})
        
        await self.keys_handler.handle(update, context, self.reserve_id)
    
    async def _download_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /download command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "download_apk", {})
        
        await self.download_handler.handle(update, context, self.reserve_id)
    
    async def _extend_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /extend command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "initiate_extend", {})
        
        return await self.extend_handler.start(update, context, self.reserve_id)
    
    async def _help_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /help command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "help", {})
        
        await self.help_handler.handle(update, context, self.reserve_id)
    
    async def _contact_command(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle /contact command"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "contact", {})
        
        # Get contact message
        if context.args:
            message = ' '.join(context.args)
            # Store support ticket
            await self.user_service.create_support_ticket(user.id, message)
            await update.message.reply_text(BotMessages.CONTACT_RECEIVED)
        else:
            await update.message.reply_text(BotMessages.CONTACT_PROMPT)
    
    async def _plan_selected(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle plan selection"""
        query = update.callback_query
        await query.answer()
        
        plan = query.data.replace("plan_", "")
        context.user_data['selected_plan'] = plan
        
        return await self.buy_handler.confirm_plan(update, context, self.reserve_id)
    
    async def _confirm_purchase(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle purchase confirmation"""
        query = update.callback_query
        await query.answer()
        
        if query.data == "confirm_yes":
            return await self.buy_handler.process_payment(update, context, self.reserve_id)
        else:
            await query.edit_message_text(BotMessages.PURCHASE_CANCELLED)
            return ConversationHandler.END
    
    async def _check_payment(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Check payment status"""
        # This would be handled by webhook in production
        # For now, just show payment instructions
        await update.message.reply_text(
            "Please complete the payment using the link sent earlier.\n"
            "Once payment is confirmed, your key will be delivered automatically."
        )
        return WAITING_PAYMENT
    
    async def _extend_key_selected(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle key selection for extension"""
        query = update.callback_query
        await query.answer()
        
        key_id = int(query.data.replace("extend_", ""))
        context.user_data['extend_key_id'] = key_id
        
        return await self.extend_handler.select_duration(update, context, self.reserve_id)
    
    async def _button_callback(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle general button callbacks"""
        query = update.callback_query
        await query.answer()
        
        data = query.data
        
        if data.startswith("btn_"):
            # Handle other button actions
            action = data.replace("btn_", "")
            
            if action == "back_to_menu":
                await self.start_handler.handle(update, context, self.reserve_id)
            elif action == "refresh_keys":
                await self.keys_handler.handle(update, context, self.reserve_id)
    
    async def _handle_message(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle regular text messages"""
        user = update.effective_user
        text = update.message.text
        
        await self.activity_tracker.track_activity(
            user.id, "message", {"text": text[:100]}
        )
        
        # Default response
        await update.message.reply_text(
            "I didn't understand that command. Please use /help to see available commands."
        )
    
    async def _handle_photo(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle photo messages"""
        user = update.effective_user
        await self.activity_tracker.track_activity(user.id, "photo_sent", {})
        
        await update.message.reply_text("Please use text commands only.")
    
    async def _cancel(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Cancel current conversation"""
        user = update.effective_user
        await update.message.reply_text(BotMessages.ACTION_CANCELLED)
        return ConversationHandler.END
    
    async def _error_handler(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Handle errors"""
        logger.error(f"Update {update} caused error {context.error}")
        
        try:
            if update and update.effective_message:
                await update.effective_message.reply_text(
                    "An error occurred. Please try again later."
                )
        except:
            pass
    
    async def start(self, webhook_url: Optional[str] = None):
        """Start the bot"""
        if webhook_url:
            # Use webhook mode
            await self.application.bot.set_webhook(
                url=f"{webhook_url}/{self.bot_token}",
                allowed_updates=Update.ALL_TYPES
            )
            logger.info(f"Bot {self.reserve_id} started with webhook: {webhook_url}")
        else:
            # Use polling mode
            await self.application.initialize()
            await self.application.start()
            await self.application.updater.start_polling()
            logger.info(f"Bot {self.reserve_id} started with polling")
    
    async def stop(self):
        """Stop the bot"""
        if self.application:
            await self.application.stop()
            await self.application.shutdown()
            logger.info(f"Bot {self.reserve_id} stopped")
    
    async def send_broadcast(self, user_ids: List[int], message: str, 
                             parse_mode: str = ParseMode.HTML):
        """Send broadcast message to users"""
        results = {
            'total': len(user_ids),
            'success': 0,
            'failed': 0,
            'failed_users': []
        }
        
        for user_id in user_ids:
            try:
                await self.application.bot.send_message(
                    chat_id=user_id,
                    text=message,
                    parse_mode=parse_mode
                )
                results['success'] += 1
            except Exception as e:
                logger.error(f"Failed to send broadcast to {user_id}: {e}")
                results['failed'] += 1
                results['failed_users'].append(user_id)
            
            # Rate limiting
            await asyncio.sleep(0.05)
        
        return results


class BotManager:
    """Manages multiple bot instances"""
    
    def __init__(self):
        self.bots: Dict[int, TelegramBot] = {}
    
    async def create_bot(self, admin_id: int, bot_token: str, 
                         reserve_id: str, db_url: str):
        """Create and initialize new bot instance"""
        if admin_id in self.bots:
            return self.bots[admin_id]
        
        bot = TelegramBot(admin_id, bot_token, reserve_id)
        await bot.initialize(db_url)
        self.bots[admin_id] = bot
        return bot
    
    async def get_bot(self, admin_id: int) -> Optional[TelegramBot]:
        """Get bot instance by admin ID"""
        return self.bots.get(admin_id)
    
    async def start_bot(self, admin_id: int, webhook_url: Optional[str] = None):
        """Start specific bot"""
        bot = self.bots.get(admin_id)
        if bot:
            await bot.start(webhook_url)
    
    async def stop_bot(self, admin_id: int):
        """Stop specific bot"""
        bot = self.bots.get(admin_id)
        if bot:
            await bot.stop()
    
    async def start_all(self, webhook_url: Optional[str] = None):
        """Start all bots"""
        for admin_id, bot in self.bots.items():
            await bot.start(webhook_url)
    
    async def stop_all(self):
        """Stop all bots"""
        for admin_id, bot in self.bots.items():
            await bot.stop()
        self.bots.clear()


# Global bot manager instance
bot_manager = BotManager()