# backend/bot_core/handlers/buy_handler.py
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
import logging
import uuid

from shared.constants.messages import BotMessages
from shared.constants import DEFAULT_PRICES
from bot_core.services.payment_processor import PaymentProcessor
from bot_core.services.key_delivery import KeyDeliveryService
from bot_core.services.activity_tracker import ActivityTracker

logger = logging.getLogger(__name__)

# Conversation states
SELECTING_PLAN, CONFIRMING_PURCHASE, WAITING_PAYMENT = range(3)

class BuyHandler:
    """Handler for /buy command and purchase flow"""
    
    def __init__(self):
        self.payment_processor = PaymentProcessor()
        self.key_delivery = KeyDeliveryService()
        self.activity_tracker = ActivityTracker()
    
    async def start(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Start purchase flow"""
        # Show available plans
        keyboard = [
            [
                InlineKeyboardButton("1 Day - ₹100", callback_data="plan_1d"),
                InlineKeyboardButton("3 Days - ₹250", callback_data="plan_3d")
            ],
            [
                InlineKeyboardButton("7 Days - ₹350", callback_data="plan_7d"),
                InlineKeyboardButton("30 Days - ₹600", callback_data="plan_30d")
            ],
            [
                InlineKeyboardButton("60 Days - ₹800", callback_data="plan_60d")
            ],
            [
                InlineKeyboardButton("❌ Cancel", callback_data="cancel")
            ]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await update.message.reply_text(
            "🛒 **Select a Plan**\n\nChoose the duration you want:",
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        return SELECTING_PLAN
    
    async def confirm_plan(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Confirm selected plan"""
        query = update.callback_query
        plan = context.user_data['selected_plan']
        
        # Get plan details
        plan_details = {
            "1d": {"name": "1 Day", "price": DEFAULT_PRICES["1d"]},
            "3d": {"name": "3 Days", "price": DEFAULT_PRICES["3d"]},
            "7d": {"name": "7 Days", "price": DEFAULT_PRICES["7d"]},
            "30d": {"name": "30 Days", "price": DEFAULT_PRICES["30d"]},
            "60d": {"name": "60 Days", "price": DEFAULT_PRICES["60d"]}
        }.get(plan)
        
        if not plan_details:
            await query.edit_message_text("Invalid plan selected.")
            return ConversationHandler.END
        
        # Store plan details
        context.user_data['plan_details'] = plan_details
        
        # Show confirmation
        keyboard = [
            [
                InlineKeyboardButton("✅ Confirm", callback_data="confirm_yes"),
                InlineKeyboardButton("❌ Cancel", callback_data="confirm_no")
            ]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            BotMessages.CONFIRM_PURCHASE.format(
                duration=plan_details['name'],
                price=plan_details['price']
            ),
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        return CONFIRMING_PURCHASE
    
    async def process_payment(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Process payment for selected plan"""
        query = update.callback_query
        user = update.effective_user
        plan_details = context.user_data['plan_details']
        
        # Generate payment link
        payment_data = await self.payment_processor.create_payment(
            user_id=user.id,
            amount=plan_details['price'],
            currency='INR',
            description=f"Purchase {plan_details['name']} Key"
        )
        
        # Store payment info
        context.user_data['payment_id'] = payment_data['payment_id']
        
        # Send payment instructions
        keyboard = [
            [InlineKeyboardButton("💳 Pay Now", url=payment_data['payment_link'])],
            [InlineKeyboardButton("✅ I've Paid", callback_data="check_payment")],
            [InlineKeyboardButton("❌ Cancel", callback_data="cancel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            BotMessages.PAYMENT_LINK.format(payment_link=payment_data['payment_link']),
            reply_markup=reply_markup,
            parse_mode='Markdown',
            disable_web_page_preview=True
        )
        
        # Track activity
        await self.activity_tracker.track_activity(
            reserve_id, user.id, 'initiate_payment',
            {'plan': plan_details['name'], 'amount': plan_details['price']}
        )
        
        return WAITING_PAYMENT
    
    async def check_payment(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Check payment status"""
        query = update.callback_query
        user = update.effective_user
        payment_id = context.user_data.get('payment_id')
        
        if not payment_id:
            await query.edit_message_text("Payment session expired. Please start over.")
            return ConversationHandler.END
        
        # Check payment status
        payment_status = await self.payment_processor.check_payment(payment_id)
        
        if payment_status['status'] == 'success':
            # Payment successful, deliver key
            key_data = await self.key_delivery.deliver_key(
                reserve_id=reserve_id,
                user_id=user.id,
                duration=context.user_data['plan_details']['name']
            )
            
            if key_data:
                # Send success message with key
                await query.edit_message_text(
                    BotMessages.PAYMENT_SUCCESS.format(
                        duration=context.user_data['plan_details']['name'],
                        key=key_data['key'],
                        expiry_date=key_data['expiry_date'],
                        days=key_data['days']
                    ),
                    parse_mode='Markdown'
                )
                
                # Track successful payment
                await self.activity_tracker.track_activity(
                    reserve_id, user.id, 'payment_success',
                    {
                        'plan': context.user_data['plan_details']['name'],
                        'amount': context.user_data['plan_details']['price'],
                        'key': key_data['key'][:8] + '...'
                    }
                )
                
                return ConversationHandler.END
            else:
                # Key delivery failed
                await query.edit_message_text(
                    "❌ Error delivering key. Please contact support."
                )
                return ConversationHandler.END
        elif payment_status['status'] == 'failed':
            # Payment failed
            await query.edit_message_text(
                BotMessages.PAYMENT_FAILED.format(reason=payment_status.get('reason', 'Unknown error'))
            )
            
            # Track failed payment
            await self.activity_tracker.track_activity(
                reserve_id, user.id, 'payment_failed',
                {
                    'plan': context.user_data['plan_details']['name'],
                    'amount': context.user_data['plan_details']['price'],
                    'reason': payment_status.get('reason')
                }
            )
            
            return ConversationHandler.END
        else:
            # Payment pending
            await query.answer("Payment not yet received. Please complete the payment.", show_alert=True)
            return WAITING_PAYMENT
    
    async def cancel(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Cancel purchase"""
        query = update.callback_query
        await query.edit_message_text(BotMessages.PURCHASE_CANCELLED)
        return ConversationHandler.END