# backend/bot_core/handlers/download_handler.py
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes
import logging
from datetime import datetime, timedelta
import secrets

from shared.constants.messages import BotMessages
from bot_core.services.key_delivery import KeyDeliveryService
from bot_core.services.activity_tracker import ActivityTracker

logger = logging.getLogger(__name__)

class DownloadHandler:
    """Handler for /download command"""
    
    def __init__(self):
        self.key_delivery = KeyDeliveryService()
        self.activity_tracker = ActivityTracker()
        self.download_links = {}  # In production, use Redis
    
    async def handle(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle /download command"""
        user = update.effective_user
        
        # Check if user has active key
        has_active_key = await self._check_active_key(reserve_id, user.id)
        
        if not has_active_key:
            # No active key
            keyboard = [
                [InlineKeyboardButton("🛒 Buy Key", callback_data="btn_buy_key")],
                [InlineKeyboardButton("❓ Help", callback_data="btn_help")]
            ]
            reply_markup = InlineKeyboardMarkup(keyboard)
            
            await update.message.reply_text(
                BotMessages.NO_ACTIVE_KEY,
                reply_markup=reply_markup,
                parse_mode='Markdown'
            )
            return
        
        # Generate temporary download link
        download_token = secrets.token_urlsafe(32)
        expiry = datetime.utcnow() + timedelta(hours=24)
        
        # Store link (in production, store in Redis)
        self.download_links[download_token] = {
            'user_id': user.id,
            'reserve_id': reserve_id,
            'expiry': expiry
        }
        
        # Generate download URL
        download_url = f"https://your-domain.com/download/{download_token}"
        
        # Send download link
        keyboard = [
            [InlineKeyboardButton("📥 Download Now", url=download_url)],
            [InlineKeyboardButton("🔙 Back to Menu", callback_data="btn_back_to_menu")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await update.message.reply_text(
            BotMessages.DOWNLOAD_PROMPT.format(download_link=download_url),
            reply_markup=reply_markup,
            parse_mode='Markdown',
            disable_web_page_preview=True
        )
        
        # Track activity
        await self.activity_tracker.track_activity(
            reserve_id, user.id, 'download_initiated',
            {'token': download_token[:8]}
        )
        
        logger.info(f"Download link generated for user {user.id} in {reserve_id}")
    
    async def serve_download(self, token: str) -> dict:
        """Serve download file (called by web endpoint)"""
        link_info = self.download_links.get(token)
        
        if not link_info:
            return {'error': 'Invalid or expired link', 'status': 404}
        
        if datetime.utcnow() > link_info['expiry']:
            del self.download_links[token]
            return {'error': 'Link expired', 'status': 410}
        
        # In production, get actual file path from database
        file_path = f"/app/uploads/apk/{link_info['reserve_id']}/current_version.apk"
        
        # Track download
        await self.activity_tracker.track_activity(
            link_info['reserve_id'],
            link_info['user_id'],
            'download_completed',
            {'token': token[:8]}
        )
        
        return {
            'file_path': file_path,
            'filename': 'app-release.apk',
            'status': 200
        }
    
    async def handle_callback(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle callback queries from download menu"""
        query = update.callback_query
        await query.answer()
        
        if query.data == "btn_buy_key":
            # Redirect to buy handler
            from bot_core.handlers.buy_handler import BuyHandler
            buy_handler = BuyHandler()
            await buy_handler.start(update, context, reserve_id)
        elif query.data == "btn_help":
            # Redirect to help handler
            from bot_core.handlers.help_handler import HelpHandler
            help_handler = HelpHandler()
            await help_handler.handle(update, context, reserve_id)
        elif query.data == "btn_back_to_menu":
            # Redirect to start
            from bot_core.handlers.start_handler import StartHandler
            start_handler = StartHandler()
            await start_handler.handle(update, context, reserve_id)
    
    async def _check_active_key(self, reserve_id: str, user_id: int) -> bool:
        """Check if user has an active key"""
        # This would query the tenant database
        # For now, return True for demo
        return True