# backend/bot_core/handlers/extend_handler.py
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes, ConversationHandler
import logging
from datetime import datetime, timedelta

from shared.constants.messages import BotMessages
from shared.constants import DEFAULT_PRICES
from bot_core.services.key_delivery import KeyDeliveryService
from bot_core.services.payment_processor import PaymentProcessor
from bot_core.services.activity_tracker import ActivityTracker

logger = logging.getLogger(__name__)

# Conversation states
SELECTING_KEY, SELECTING_DURATION, CONFIRMING_EXTEND = range(3, 6)

class ExtendHandler:
    """Handler for /extend command"""
    
    def __init__(self):
        self.key_delivery = KeyDeliveryService()
        self.payment_processor = PaymentProcessor()
        self.activity_tracker = ActivityTracker()
    
    async def start(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Start key extension flow"""
        user = update.effective_user
        
        # Get user's keys that can be extended
        keys = await self._get_extendable_keys(reserve_id, user.id)
        
        if not keys:
            await update.message.reply_text(
                "You don't have any keys that can be extended.\n"
                "Use /buy to purchase a new key."
            )
            return ConversationHandler.END
        
        # Build keyboard with keys
        keyboard = []
        for key in keys[:5]:  # Limit to 5 keys
            keyboard.append([
                InlineKeyboardButton(
                    f"{key['key'][:8]}... ({key['days_left']} days left)",
                    callback_data=f"extend_key_{key['id']}"
                )
            ])
        
        keyboard.append([InlineKeyboardButton("❌ Cancel", callback_data="cancel")])
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await update.message.reply_text(
            "🔑 **Select a key to extend:**\n\n"
            "Choose from your active keys below:",
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        return SELECTING_KEY
    
    async def select_duration(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Select duration for extension"""
        query = update.callback_query
        await query.answer()
        
        key_id = int(query.data.replace("extend_key_", ""))
        context.user_data['extend_key_id'] = key_id
        
        # Show duration options
        keyboard = [
            [
                InlineKeyboardButton("7 Days - ₹350", callback_data="extend_dur_7d"),
                InlineKeyboardButton("30 Days - ₹600", callback_data="extend_dur_30d")
            ],
            [
                InlineKeyboardButton("60 Days - ₹800", callback_data="extend_dur_60d")
            ],
            [InlineKeyboardButton("❌ Cancel", callback_data="cancel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            "⏰ **Select extension duration:**\n\n"
            "How many days would you like to add?",
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        return SELECTING_DURATION
    
    async def confirm_extend(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Confirm key extension"""
        query = update.callback_query
        await query.answer()
        
        duration = query.data.replace("extend_dur_", "")
        context.user_data['extend_duration'] = duration
        
        # Get price for duration
        prices = {
            '7d': 350,
            '30d': 600,
            '60d': 800
        }
        price = prices.get(duration, 350)
        
        # Show confirmation
        keyboard = [
            [
                InlineKeyboardButton("✅ Confirm", callback_data="extend_confirm_yes"),
                InlineKeyboardButton("❌ Cancel", callback_data="cancel")
            ]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            f"**Confirm Extension**\n\n"
            f"Add {duration.replace('d', ' days')}: ₹{price}\n\n"
            f"Proceed with payment?",
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        return CONFIRMING_EXTEND
    
    async def process_extension(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Process payment for key extension"""
        query = update.callback_query
        user = update.effective_user
        
        key_id = context.user_data['extend_key_id']
        duration = context.user_data['extend_duration']
        
        prices = {
            '7d': 350,
            '30d': 600,
            '60d': 800
        }
        price = prices.get(duration, 350)
        
        # Create payment
        payment_data = await self.payment_processor.create_payment(
            user_id=user.id,
            amount=price,
            currency='INR',
            description=f"Extend key by {duration}"
        )
        
        # Store payment info
        context.user_data['extension_payment_id'] = payment_data['payment_id']
        
        # Send payment link
        keyboard = [
            [InlineKeyboardButton("💳 Pay Now", url=payment_data['payment_link'])],
            [InlineKeyboardButton("✅ I've Paid", callback_data="extend_check_payment")],
            [InlineKeyboardButton("❌ Cancel", callback_data="cancel")]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await query.edit_message_text(
            f"**Payment Required**\n\n"
            f"Amount: ₹{price}\n\n"
            f"Click below to complete payment:",
            reply_markup=reply_markup,
            parse_mode='Markdown',
            disable_web_page_preview=True
        )
        
        # Track activity
        await self.activity_tracker.track_activity(
            reserve_id, user.id, 'initiate_extension',
            {'key_id': key_id, 'duration': duration, 'amount': price}
        )
        
        return CONFIRMING_EXTEND
    
    async def check_payment(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Check payment status for extension"""
        query = update.callback_query
        user = update.effective_user
        
        payment_id = context.user_data.get('extension_payment_id')
        
        if not payment_id:
            await query.edit_message_text("Payment session expired. Please start over.")
            return ConversationHandler.END
        
        # Check payment status
        payment_status = await self.payment_processor.check_payment(payment_id)
        
        if payment_status['status'] == 'success':
            # Extend the key
            key_id = context.user_data['extend_key_id']
            duration = context.user_data['extend_duration']
            days = int(duration.replace('d', ''))
            
            result = await self.key_delivery.extend_key(
                reserve_id=reserve_id,
                key_id=key_id,
                days=days
            )
            
            if result:
                await query.edit_message_text(
                    f"✅ **Key Extended Successfully!**\n\n"
                    f"Key: {result['key'][:8]}...\n"
                    f"New Expiry: {result['new_expiry']}\n"
                    f"Days Added: {days}",
                    parse_mode='Markdown'
                )
                
                # Track successful extension
                await self.activity_tracker.track_activity(
                    reserve_id, user.id, 'extension_success',
                    {'key_id': key_id, 'duration': duration, 'days': days}
                )
            else:
                await query.edit_message_text(
                    "❌ Error extending key. Please contact support."
                )
            
            return ConversationHandler.END
        elif payment_status['status'] == 'failed':
            await query.edit_message_text(
                f"❌ Payment failed: {payment_status.get('reason', 'Unknown error')}"
            )
            return ConversationHandler.END
        else:
            await query.answer("Payment not yet received. Please complete the payment.", show_alert=True)
            return CONFIRMING_EXTEND
    
    async def cancel(self, update: Update, context: ContextTypes.DEFAULT_TYPE):
        """Cancel extension"""
        query = update.callback_query
        await query.edit_message_text("❌ Extension cancelled.")
        return ConversationHandler.END
    
    async def _get_extendable_keys(self, reserve_id: str, user_id: int) -> list:
        """Get user's keys that can be extended"""
        # This would query the tenant database
        # For now, return mock data
        now = datetime.utcnow()
        
        return [
            {
                'id': 1,
                'key': 'GAME-ABCD-1234-EFGH-5678',
                'days_left': 28,
                'expires_at': now + timedelta(days=28)
            },
            {
                'id': 2,
                'key': 'GAME-WXYZ-9876-PQRS-5432',
                'days_left': 2,
                'expires_at': now + timedelta(days=2)
            }
        ]