# backend/bot_core/handlers/help_handler.py
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes
import logging

from shared.constants.messages import BotMessages
from bot_core.services.activity_tracker import ActivityTracker

logger = logging.getLogger(__name__)

class HelpHandler:
    """Handler for /help command"""
    
    def __init__(self):
        self.activity_tracker = ActivityTracker()
    
    async def handle(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle /help command"""
        user = update.effective_user
        
        # Get bot-specific help (could be customized per bot)
        help_text = BotMessages.HELP
        
        # Add support contact
        help_text += "\n\n**Support Contact:**\n"
        help_text += "📧 Email: support@yourbot.com\n"
        help_text += "💬 Telegram: @support_bot\n"
        help_text += "⏰ Response Time: < 2 hours"
        
        # Create keyboard with quick actions
        keyboard = [
            [
                InlineKeyboardButton("📋 Plans", callback_data="btn_plans"),
                InlineKeyboardButton("🛒 Buy", callback_data="btn_buy")
            ],
            [
                InlineKeyboardButton("🔑 My Keys", callback_data="btn_my_keys"),
                InlineKeyboardButton("📱 Download", callback_data="btn_download")
            ],
            [
                InlineKeyboardButton("📧 Contact Support", url="https://t.me/support_bot")
            ]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await update.message.reply_text(
            help_text,
            reply_markup=reply_markup,
            parse_mode='Markdown',
            disable_web_page_preview=True
        )
        
        # Track activity
        await self.activity_tracker.track_activity(
            reserve_id, user.id, 'help_viewed',
            {}
        )
    
    async def handle_callback(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle callback queries from help menu"""
        query = update.callback_query
        await query.answer()
        
        if query.data == "btn_plans":
            from bot_core.handlers.plans_handler import PlansHandler
            handler = PlansHandler()
            await handler.handle(update, context, reserve_id)
        elif query.data == "btn_buy":
            from bot_core.handlers.buy_handler import BuyHandler
            handler = BuyHandler()
            await handler.start(update, context, reserve_id)
        elif query.data == "btn_my_keys":
            from bot_core.handlers.keys_handler import KeysHandler
            handler = KeysHandler()
            await handler.handle(update, context, reserve_id)
        elif query.data == "btn_download":
            from bot_core.handlers.download_handler import DownloadHandler
            handler = DownloadHandler()
            await handler.handle(update, context, reserve_id)