# backend/bot_core/handlers/keys_handler.py
from telegram import Update, InlineKeyboardButton, InlineKeyboardMarkup
from telegram.ext import ContextTypes
import logging
from datetime import datetime, timedelta

from shared.constants.messages import BotMessages
from bot_core.services.key_delivery import KeyDeliveryService
from bot_core.services.activity_tracker import ActivityTracker

logger = logging.getLogger(__name__)

class KeysHandler:
    """Handler for /my_keys command"""
    
    def __init__(self):
        self.key_delivery = KeyDeliveryService()
        self.activity_tracker = ActivityTracker()
    
    async def handle(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle /my_keys command"""
        user = update.effective_user
        
        # Get user's keys from database
        keys = await self._get_user_keys(reserve_id, user.id)
        
        if not keys:
            await update.message.reply_text(
                BotMessages.NO_KEYS,
                parse_mode='Markdown'
            )
            return
        
        # Build keys message
        message = BotMessages.KEYS_HEADER + "\n"
        
        for i, key in enumerate(keys, 1):
            status_emoji = "✅" if key['status'] == 'active' else "⚠️"
            
            message += BotMessages.KEY_DETAIL.format(
                index=i,
                key=key['key'][:8] + "..." + key['key'][-4:],  # Show partial key for security
                duration=key['duration'],
                purchase_date=key['purchased_at'].strftime('%d %b %Y'),
                expiry_date=key['expires_at'].strftime('%d %b %Y'),
                days_left=key['days_left'],
                status=f"{status_emoji} Active" if key['status'] == 'active' else "⚠️ Expiring Soon"
            )
        
        # Add action buttons
        keyboard = [
            [
                InlineKeyboardButton("🔄 Refresh", callback_data="btn_refresh_keys"),
                InlineKeyboardButton("🛒 Buy More", callback_data="btn_buy_more")
            ],
            [
                InlineKeyboardButton("📱 Download App", callback_data="btn_download")
            ]
        ]
        reply_markup = InlineKeyboardMarkup(keyboard)
        
        await update.message.reply_text(
            message,
            reply_markup=reply_markup,
            parse_mode='Markdown'
        )
        
        # Track activity
        await self.activity_tracker.track_activity(
            reserve_id, user.id, 'view_keys',
            {'key_count': len(keys)}
        )
    
    async def handle_callback(self, update: Update, context: ContextTypes.DEFAULT_TYPE, reserve_id: str):
        """Handle callback queries from keys menu"""
        query = update.callback_query
        await query.answer()
        
        if query.data == "btn_refresh_keys":
            await self.handle(update, context, reserve_id)
        elif query.data == "btn_buy_more":
            # Redirect to buy handler
            from bot_core.handlers.buy_handler import BuyHandler
            buy_handler = BuyHandler()
            await buy_handler.start(update, context, reserve_id)
        elif query.data == "btn_download":
            # Redirect to download handler
            from bot_core.handlers.download_handler import DownloadHandler
            download_handler = DownloadHandler()
            await download_handler.handle(update, context, reserve_id)
    
    async def _get_user_keys(self, reserve_id: str, user_id: int) -> list:
        """Get user's keys from database"""
        # This would query the tenant database
        # For now, return mock data
        now = datetime.utcnow()
        
        return [
            {
                'key': 'GAME-ABCD-1234-EFGH-5678',
                'duration': '30 Days',
                'purchased_at': now - timedelta(days=2),
                'expires_at': now + timedelta(days=28),
                'days_left': 28,
                'status': 'active'
            },
            {
                'key': 'GAME-WXYZ-9876-PQRS-5432',
                'duration': '7 Days',
                'purchased_at': now - timedelta(days=5),
                'expires_at': now + timedelta(days=2),
                'days_left': 2,
                'status': 'expiring'
            }
        ]