# backend/bot_core/services/activity_tracker.py
from typing import Dict, Any, Optional
import logging
from datetime import datetime

from core.database import tenant_db_manager

logger = logging.getLogger(__name__)

class ActivityTracker:
    """Service for tracking user activities"""
    
    def __init__(self, reserve_id: str):
        self.reserve_id = reserve_id
        self.db = None
    
    async def initialize(self, db_url: str):
        """Initialize with database connection"""
        self.db = tenant_db_manager.get_tenant_session(self.reserve_id, db_url)
    
    async def track_activity(self, user_id: int, activity_type: str,
                            details: Dict[str, Any], ip_address: Optional[str] = None):
        """Track user activity"""
        # This would insert into tenant database
        activity = {
            'user_id': user_id,
            'activity_type': activity_type,
            'details': details,
            'ip_address': ip_address,
            'created_at': datetime.utcnow()
        }
        
        logger.info(f"Activity tracked for user {user_id}: {activity_type}")
        
        # In production, save to database
        # await self.db.execute(...)
        
        return activity
    
    async def track_session_start(self, user_id: int):
        """Track session start"""
        return await self.track_activity(user_id, 'session_start', {})
    
    async def track_session_end(self, user_id: int, duration: int,
                               commands_used: list):
        """Track session end"""
        return await self.track_activity(
            user_id, 'session_end',
            {'duration': duration, 'commands': commands_used}
        )
    
    async def get_user_activity(self, user_id: int, limit: int = 50) -> list:
        """Get user's recent activity"""
        # This would query tenant database
        return []