# backend/bot_core/services/key_delivery.py
from typing import Dict, Any, Optional
import logging
from datetime import datetime, timedelta
import random

from core.utils import key_generator, datetime_utils

logger = logging.getLogger(__name__)

class KeyDeliveryService:
    """Service for delivering keys to users"""
    
    async def deliver_key(self, reserve_id: str, user_id: int, 
                         duration: str) -> Optional[Dict[str, Any]]:
        """Deliver key to user after successful payment"""
        # Try to get stored key first
        key = await self._get_stored_key(reserve_id, duration)
        
        if not key:
            # Generate new key if auto-generation enabled
            key = await self._generate_key(reserve_id, duration)
        
        if key:
            # Assign key to user
            assigned_key = await self._assign_key_to_user(
                reserve_id, key['key_value'], user_id, duration
            )
            
            if assigned_key:
                return assigned_key
        
        return None
    
    async def extend_key(self, reserve_id: str, key_id: int, 
                        days: int) -> Optional[Dict[str, Any]]:
        """Extend existing key"""
        # This would update tenant database
        new_expiry = datetime_utils.get_expiry_date(days)
        
        return {
            'key_id': key_id,
            'key': 'GAME-ABCD-1234',
            'new_expiry': new_expiry.strftime('%d %b %Y'),
            'days': days
        }
    
    async def validate_key(self, reserve_id: str, key_value: str) -> Dict[str, Any]:
        """Validate a key"""
        # This would check tenant database
        return {
            'valid': True,
            'key': key_value,
            'duration': '30d',
            'expires_at': (datetime.utcnow() + timedelta(days=30)).isoformat()
        }
    
    async def _get_stored_key(self, reserve_id: str, duration: str) -> Optional[Dict[str, Any]]:
        """Get available stored key"""
        # This would query tenant database
        # For now, return mock data
        if random.random() > 0.3:  # 70% chance of having stored key
            return {
                'key_value': f'STORED-{duration}-{random.randint(1000, 9999)}',
                'duration': duration,
                'price': 600
            }
        return None
    
    async def _generate_key(self, reserve_id: str, duration: str) -> Optional[Dict[str, Any]]:
        """Generate new key"""
        # Get key generation pattern from tenant settings
        pattern = "{PREFIX}-{YYYYMMDD}-{RANDOM8}"
        
        key_value = key_generator.generate_key(
            pattern=pattern,
            prefix='GAME',
            exclude_similar=True,
            enable_checksum=False
        )
        
        return {
            'key_value': key_value,
            'duration': duration,
            'price': 600
        }
    
    async def _assign_key_to_user(self, reserve_id: str, key_value: str,
                                 user_id: int, duration: str) -> Dict[str, Any]:
        """Assign key to user in database"""
        # This would update tenant database
        expiry_date = datetime_utils.get_expiry_date(int(duration.replace('d', '')))
        
        return {
            'key': key_value,
            'duration': duration,
            'expiry_date': expiry_date.strftime('%d %b %Y'),
            'days': int(duration.replace('d', ''))
        }