# backend/config/constants.py
from enum import Enum
from typing import Dict, List, Any

class UserRole(str, Enum):
    """User roles in the system"""
    OWNER = "owner"
    ADMIN = "admin"
    USER = "user"


class AdminStatus(str, Enum):
    """Admin account status"""
    ACTIVE = "active"
    INACTIVE = "inactive"
    BLOCKED = "blocked"
    EXPIRED = "expired"


class RechargeStatus(str, Enum):
    """Recharge status for admin panels"""
    ACTIVE = "active"
    EXPIRED = "expired"
    PENDING = "pending"
    CANCELLED = "cancelled"


class PaymentGateway(str, Enum):
    """Supported payment gateways"""
    RAZORPAY = "razorpay"
    PHONEPE = "phonepe"
    PAYTM = "paytm"
    STRIPE = "stripe"
    PAYPAL = "paypal"


class KeyType(str, Enum):
    """Types of keys"""
    STORED = "stored"
    GENERATED = "generated"


class KeyStatus(str, Enum):
    """Key status values"""
    AVAILABLE = "available"
    SOLD = "sold"
    BLOCKED = "blocked"
    EXPIRED = "expired"


class KeyDuration(str, Enum):
    """Key duration options"""
    DAY_1 = "1d"
    DAY_3 = "3d"
    DAY_7 = "7d"
    DAY_30 = "30d"
    DAY_60 = "60d"


class ActivityType(str, Enum):
    """Types of user activities"""
    START = "start"
    VIEW_PLANS = "view_plans"
    INITIATE_PURCHASE = "initiate_purchase"
    PAYMENT_SUCCESS = "payment_success"
    PAYMENT_FAILED = "payment_failed"
    KEY_RECEIVED = "key_received"
    DOWNLOAD = "download"
    VIEW_KEYS = "view_keys"
    EXTEND = "extend"
    REDEEM = "redeem"
    HELP = "help"
    CONTACT = "contact"
    BROADCAST_OPEN = "broadcast_open"
    SESSION_START = "session_start"
    SESSION_END = "session_end"


class AdminActivityType(str, Enum):
    """Types of admin activities"""
    LOGIN = "login"
    LOGOUT = "logout"
    KEY_GENERATED = "key_generated"
    KEY_UPLOADED = "key_uploaded"
    KEY_PATTERN_CHANGED = "key_pattern_changed"
    PRICE_UPDATED = "price_updated"
    BROADCAST_SENT = "broadcast_sent"
    USER_BLOCKED = "user_blocked"
    USER_UNBLOCKED = "user_unblocked"
    APK_UPLOADED = "apk_uploaded"
    PAYMENT_CONFIGURED = "payment_configured"
    SETTINGS_MODIFIED = "settings_modified"
    REFUND_PROCESSED = "refund_processed"


class TransactionStatus(str, Enum):
    """Payment transaction status"""
    PENDING = "pending"
    SUCCESS = "success"
    FAILED = "failed"
    REFUNDED = "refunded"


class BroadcastStatus(str, Enum):
    """Broadcast delivery status"""
    DRAFT = "draft"
    SCHEDULED = "scheduled"
    SENDING = "sending"
    COMPLETED = "completed"
    FAILED = "failed"
    CANCELLED = "cancelled"


class BroadcastTarget(str, Enum):
    """Broadcast target types"""
    ALL = "all"
    PREMIUM = "premium"
    ACTIVE = "active"
    INACTIVE = "inactive"
    CUSTOM = "custom"


class ServerStatus(str, Enum):
    """Server health status"""
    ONLINE = "online"
    OFFLINE = "offline"
    DEGRADED = "degraded"
    MAINTENANCE = "maintenance"


class AlertSeverity(str, Enum):
    """Alert severity levels"""
    INFO = "info"
    WARNING = "warning"
    CRITICAL = "critical"
    EMERGENCY = "emergency"


class NotificationChannel(str, Enum):
    """Notification channels"""
    TELEGRAM = "telegram"
    EMAIL = "email"
    SMS = "sms"
    WEBHOOK = "webhook"


# Price configuration
DEFAULT_PRICES: Dict[str, float] = {
    "1d": 100,
    "3d": 250,
    "7d": 350,
    "30d": 600,
    "60d": 800
}

PRICE_RANGES: Dict[str, Dict[str, float]] = {
    "1d": {"min": 80, "max": 120},
    "3d": {"min": 200, "max": 300},
    "7d": {"min": 300, "max": 400},
    "30d": {"min": 500, "max": 700},
    "60d": {"min": 700, "max": 900}
}

# Bulk discount configuration
BULK_DISCOUNTS: List[Dict[str, Any]] = [
    {"min_keys": 10, "max_keys": 50, "discount": 5},
    {"min_keys": 51, "max_keys": 200, "discount": 10},
    {"min_keys": 201, "max_keys": 10000, "discount": 15}
]

# Recharge plans
RECHARGE_PLANS: Dict[str, Dict[str, Any]] = {
    "monthly": {
        "name": "Monthly",
        "days": 30,
        "price": 200,
        "key_limit": 1000,
        "features": ["1000 keys/month", "Full access", "Basic support"]
    },
    "6months": {
        "name": "6 Months",
        "days": 180,
        "price": 900,
        "key_limit": 6000,
        "features": ["6000 keys total", "Priority support", "Save ₹300"]
    },
    "yearly": {
        "name": "Yearly",
        "days": 365,
        "price": 1700,
        "key_limit": 12000,
        "features": ["12000 keys total", "Premium support", "Save ₹700"]
    }
}

# Key generation patterns
KEY_PATTERN_VARIABLES: Dict[str, str] = {
    "{DATE}": "Current date (YYYYMMDD)",
    "{YYYYMMDD}": "Year-Month-Day (20240115)",
    "{DDMMYYYY}": "Day-Month-Year (15012024)",
    "{RANDOM4}": "4 random characters",
    "{RANDOM6}": "6 random characters",
    "{RANDOM8}": "8 random characters",
    "{RANDOM10}": "10 random characters",
    "{PREFIX}": "Custom prefix",
    "{CHECKSUM}": "Auto-checksum digit"
}

# Default key generation pattern
DEFAULT_KEY_PATTERN = "{PREFIX}-{YYYYMMDD}-{RANDOM8}"

# Activity tracking retention (days)
ACTIVITY_RETENTION_DAYS = 90

# Cache TTL values (seconds)
CACHE_TTL = {
    "user": 3600,
    "key": 600,
    "plan": 3600,
    "admin": 300,
    "stats": 300
}

# Rate limits (requests per minute)
RATE_LIMITS = {
    "default": 100,
    "auth": 10,
    "payment": 20,
    "broadcast": 5,
    "key_generation": 10,
    "api": 60
}

# File upload limits (bytes)
MAX_FILE_SIZES = {
    "apk": 100 * 1024 * 1024,  # 100MB
    "csv": 50 * 1024 * 1024,   # 50MB
    "image": 5 * 1024 * 1024    # 5MB
}

# Allowed file extensions
ALLOWED_EXTENSIONS = {
    "apk": [".apk"],
    "csv": [".csv", ".xlsx", ".xls"],
    "image": [".jpg", ".jpeg", ".png", ".gif"]
}

# Pagination defaults
PAGINATION = {
    "default_limit": 50,
    "max_limit": 1000
}

# Webhook events
WEBHOOK_EVENTS = [
    "payment.success",
    "payment.failed",
    "key.generated",
    "key.sold",
    "user.joined",
    "broadcast.sent",
    "alert.triggered"
]