# backend/config/settings.py
import os
from typing import Optional, Dict, Any, List
from pydantic_settings import BaseSettings
from pydantic import PostgresDsn, validator, AnyHttpUrl, Field
from enum import Enum


class Environment(str, Enum):
    DEVELOPMENT = "development"
    STAGING = "staging"
    PRODUCTION = "production"


class PaymentGateway(str, Enum):
    RAZORPAY = "razorpay"
    PHONEPE = "phonepe"
    PAYTM = "paytm"
    STRIPE = "stripe"
    PAYPAL = "paypal"


class Settings(BaseSettings):
    # Application Settings
    APP_NAME: str = "Telegram Multi-Tenant Bot System"
    APP_VERSION: str = "1.0.0"
    ENVIRONMENT: Environment = Environment.DEVELOPMENT
    DEBUG: bool = False
    SECRET_KEY: str
    JWT_SECRET_KEY: str
    JWT_ALGORITHM: str = "HS256"
    JWT_ACCESS_TOKEN_EXPIRE_MINUTES: int = 30
    JWT_REFRESH_TOKEN_EXPIRE_DAYS: int = 7

    # Database Settings
    DATABASE_URL: PostgresDsn
    DATABASE_POOL_SIZE: int = 20
    DATABASE_MAX_OVERFLOW: int = 10
    DATABASE_POOL_TIMEOUT: int = 30
    DATABASE_ECHO: bool = False
    DATABASE_POOL_PRE_PING: bool = True

    # Redis Settings
    REDIS_URL: str
    REDIS_MAX_CONNECTIONS: int = 10
    REDIS_SOCKET_TIMEOUT: int = 5
    REDIS_SOCKET_CONNECT_TIMEOUT: int = 5

    # RabbitMQ Settings
    RABBITMQ_URL: str
    RABBITMQ_EXCHANGE: str = "telegram_bot"
    RABBITMQ_QUEUE_PREFIX: str = "tenant_"

    # Celery Settings
    CELERY_BROKER_URL: str
    CELERY_RESULT_BACKEND: str
    CELERY_TASK_SERIALIZER: str = "json"
    CELERY_RESULT_SERIALIZER: str = "json"
    CELERY_ACCEPT_CONTENT: List[str] = ["json"]
    CELERY_TIMEZONE: str = "UTC"
    CELERY_ENABLE_UTC: bool = True
    CELERY_TASK_TRACK_STARTED: bool = True
    CELERY_TASK_TIME_LIMIT: int = 30 * 60
    CELERY_TASK_SOFT_TIME_LIMIT: int = 25 * 60

    # MinIO Settings
    MINIO_ENDPOINT: str = "minio:9000"
    MINIO_ACCESS_KEY: str
    MINIO_SECRET_KEY: str
    MINIO_SECURE: bool = False
    MINIO_BUCKET_APK: str = "apk-files"
    MINIO_BUCKET_BACKUPS: str = "backups"

    # Telegram Settings
    TELEGRAM_API_ID: Optional[int] = None
    TELEGRAM_API_HASH: Optional[str] = None
    TELEGRAM_BOT_TOKEN: Optional[str] = None
    TELEGRAM_WEBHOOK_URL: Optional[AnyHttpUrl] = None
    TELEGRAM_WEBHOOK_SECRET: Optional[str] = None
    TELEGRAM_MAX_CONNECTIONS: int = 100

    # Payment Gateway Settings
    DEFAULT_PAYMENT_GATEWAY: PaymentGateway = PaymentGateway.RAZORPAY
    RAZORPAY_KEY_ID: Optional[str] = None
    RAZORPAY_KEY_SECRET: Optional[str] = None
    RAZORPAY_WEBHOOK_SECRET: Optional[str] = None

    STRIPE_PUBLIC_KEY: Optional[str] = None
    STRIPE_SECRET_KEY: Optional[str] = None
    STRIPE_WEBHOOK_SECRET: Optional[str] = None

    PAYPAL_CLIENT_ID: Optional[str] = None
    PAYPAL_CLIENT_SECRET: Optional[str] = None
    PAYPAL_MODE: str = "sandbox"

    PHONEPE_MERCHANT_ID: Optional[str] = None
    PHONEPE_SALT_KEY: Optional[str] = None
    PHONEPE_SALT_INDEX: Optional[int] = 1
    PHONEPE_ENV: str = "UAT"

    # Security Settings
    CORS_ORIGINS: List[str] = ["http://localhost:3000", "http://localhost:8000"]
    CORS_ALLOW_CREDENTIALS: bool = True
    CORS_ALLOW_METHODS: List[str] = ["*"]
    CORS_ALLOW_HEADERS: List[str] = ["*"]

    RATE_LIMIT_ENABLED: bool = True
    RATE_LIMIT_DEFAULT: str = "100/minute"
    RATE_LIMIT_STRICT: str = "10/minute"

    # File Upload Settings
    MAX_UPLOAD_SIZE: int = 100 * 1024 * 1024  # 100MB
    ALLOWED_EXTENSIONS: List[str] = [".apk", ".zip", ".csv", ".xlsx"]
    APK_STORAGE_PATH: str = "/app/uploads/apk"
    KEY_STORAGE_PATH: str = "/app/uploads/keys"

    # Activity Tracking
    TRACK_USER_ACTIVITY: bool = True
    TRACK_ADMIN_ACTIVITY: bool = True
    ACTIVITY_RETENTION_DAYS: int = 90
    SESSION_RETENTION_DAYS: int = 30

    # Key Generation Settings
    KEY_MIN_LENGTH: int = 8
    KEY_MAX_LENGTH: int = 50
    KEY_DEFAULT_PATTERN: str = "{PREFIX}-{YYYYMMDD}-{RANDOM8}"
    KEY_EXCLUDE_SIMILAR: bool = True
    KEY_CHECKSUM_ENABLED: bool = False
    KEY_BATCH_SIZE: int = 1000
    KEY_LOW_STOCK_ALERT: int = 100

    # Broadcast Settings
    BROADCAST_BATCH_SIZE: int = 100
    BROADCAST_MAX_RETRIES: int = 3
    BROADCAST_RETRY_DELAY: int = 60  # seconds

    # Monitoring Settings
    PROMETHEUS_ENABLED: bool = True
    SENTRY_DSN: Optional[str] = None
    SENTRY_ENVIRONMENT: Optional[str] = None
    SENTRY_TRACES_SAMPLE_RATE: float = 1.0

    # Logging Settings
    LOG_LEVEL: str = "INFO"
    LOG_FORMAT: str = "json"
    LOG_FILE: Optional[str] = "/app/logs/app.log"
    LOG_RETENTION_DAYS: int = 30

    # Email Settings
    SMTP_HOST: Optional[str] = None
    SMTP_PORT: Optional[int] = 587
    SMTP_USER: Optional[str] = None
    SMTP_PASSWORD: Optional[str] = None
    SMTP_USE_TLS: bool = True
    EMAIL_FROM: Optional[str] = None
    EMAIL_FROM_NAME: Optional[str] = "Telegram Bot System"

    # Backup Settings
    BACKUP_ENABLED: bool = True
    BACKUP_SCHEDULE: str = "0 2 * * *"  # 2 AM daily
    BACKUP_RETENTION_DAYS: int = 30
    BACKUP_PATH: str = "/app/backups"

    # Cache Settings
    CACHE_TTL_DEFAULT: int = 300  # 5 minutes
    CACHE_TTL_USER: int = 3600  # 1 hour
    CACHE_TTL_KEY: int = 600  # 10 minutes
    CACHE_TTL_PLAN: int = 3600  # 1 hour

    @validator("DATABASE_URL", pre=True)
    def assemble_db_connection(cls, v: Optional[str], values: Dict[str, Any]) -> str:
        if isinstance(v, str):
            return v
        return PostgresDsn.build(
            scheme="postgresql",
            username=values.get("DB_USER"),
            password=values.get("DB_PASSWORD"),
            host=values.get("DB_HOST"),
            port=values.get("DB_PORT"),
            path=f"{values.get('DB_NAME') or ''}",
        )

    @validator("CELERY_BROKER_URL", pre=True)
    def assemble_celery_broker(cls, v: Optional[str], values: Dict[str, Any]) -> str:
        if isinstance(v, str):
            return v
        return values.get("RABBITMQ_URL", "")

    @validator("CELERY_RESULT_BACKEND", pre=True)
    def assemble_celery_backend(cls, v: Optional[str], values: Dict[str, Any]) -> str:
        if isinstance(v, str):
            return v
        return values.get("REDIS_URL", "")

    class Config:
        env_file = ".env"
        env_file_encoding = "utf-8"
        case_sensitive = True


# Create global settings instance
settings = Settings()


# Environment specific settings
class DevelopmentSettings(Settings):
    DEBUG: bool = True
    LOG_LEVEL: str = "DEBUG"


class ProductionSettings(Settings):
    DEBUG: bool = False
    LOG_LEVEL: str = "WARNING"
    CORS_ORIGINS: List[str] = []


def get_settings() -> Settings:
    env = os.getenv("ENVIRONMENT", "development")
    if env == "production":
        return ProductionSettings()
    return DevelopmentSettings()