# backend/core/database.py
from sqlalchemy import create_engine, MetaData
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import sessionmaker, Session
from sqlalchemy.pool import QueuePool
from typing import Generator, Optional, Dict, Any
import redis.asyncio as redis
from contextlib import contextmanager
import logging

from config.settings import settings

logger = logging.getLogger(__name__)

# Create SQLAlchemy engine
engine = create_engine(
    str(settings.DATABASE_URL),
    pool_size=settings.DATABASE_POOL_SIZE,
    max_overflow=settings.DATABASE_MAX_OVERFLOW,
    pool_timeout=settings.DATABASE_POOL_TIMEOUT,
    pool_pre_ping=settings.DATABASE_POOL_PRE_PING,
    echo=settings.DATABASE_ECHO,
    poolclass=QueuePool,
)

# Create session factory
SessionLocal = sessionmaker(autocommit=False, autoflush=False, bind=engine)

# Create base class for models
Base = declarative_base()
metadata = MetaData()


# Database session dependency
def get_db() -> Generator[Session, None, None]:
    db = SessionLocal()
    try:
        yield db
    finally:
        db.close()


# Async Redis client
async def get_redis_client() -> redis.Redis:
    return redis.from_url(
        settings.REDIS_URL,
        max_connections=settings.REDIS_MAX_CONNECTIONS,
        socket_timeout=settings.REDIS_SOCKET_TIMEOUT,
        socket_connect_timeout=settings.REDIS_SOCKET_CONNECT_TIMEOUT,
        decode_responses=True,
    )


# Context manager for database sessions
@contextmanager
def db_session() -> Generator[Session, None, None]:
    session = SessionLocal()
    try:
        yield session
        session.commit()
    except Exception as e:
        session.rollback()
        logger.error(f"Database error: {e}")
        raise
    finally:
        session.close()


# Tenant database connection manager
class TenantDatabaseManager:
    """Manages connections to tenant-specific databases"""

    def __init__(self):
        self._engines: Dict[str, Any] = {}
        self._sessions: Dict[str, Any] = {}

    def get_tenant_engine(self, tenant_id: str, db_url: str):
        """Get or create engine for tenant database"""
        if tenant_id not in self._engines:
            self._engines[tenant_id] = create_engine(
                db_url,
                pool_size=5,
                max_overflow=10,
                pool_pre_ping=True,
                poolclass=QueuePool,
            )
        return self._engines[tenant_id]

    def get_tenant_session(self, tenant_id: str, db_url: str) -> Session:
        """Get session for tenant database"""
        if tenant_id not in self._sessions:
            engine = self.get_tenant_engine(tenant_id, db_url)
            session_factory = sessionmaker(bind=engine)
            self._sessions[tenant_id] = session_factory
        return self._sessions[tenant_id]()

    def close_all(self):
        """Close all tenant connections"""
        for engine in self._engines.values():
            engine.dispose()
        self._engines.clear()
        self._sessions.clear()


tenant_db_manager = TenantDatabaseManager()


# Health check function
async def check_database_health() -> bool:
    """Check if database is healthy"""
    try:
        with db_session() as session:
            session.execute("SELECT 1")
        return True
    except Exception as e:
        logger.error(f"Database health check failed: {e}")
        return False


# Initialize database
def init_database():
    """Initialize database tables"""
    try:
        Base.metadata.create_all(bind=engine)
        logger.info("Database initialized successfully")
    except Exception as e:
        logger.error(f"Database initialization failed: {e}")
        raise


# Alembic migrations helper
def run_migrations():
    """Run database migrations"""
    from alembic.config import Config
    from alembic import command

    alembic_cfg = Config("alembic.ini")
    command.upgrade(alembic_cfg, "head")