# backend/core/exceptions.py
from typing import Any, Optional, Dict

class BaseAppException(Exception):
    """Base exception for application"""
    
    def __init__(
        self,
        message: str,
        status_code: int = 500,
        error_code: Optional[str] = None,
        details: Optional[Dict[str, Any]] = None
    ):
        self.message = message
        self.status_code = status_code
        self.error_code = error_code
        self.details = details or {}
        super().__init__(self.message)


class NotFoundError(BaseAppException):
    """Resource not found exception"""
    
    def __init__(self, message: str = "Resource not found", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=404,
            error_code="NOT_FOUND",
            details=details
        )


class ValidationError(BaseAppException):
    """Validation error exception"""
    
    def __init__(self, message: str = "Validation error", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=400,
            error_code="VALIDATION_ERROR",
            details=details
        )


class AuthenticationError(BaseAppException):
    """Authentication error exception"""
    
    def __init__(self, message: str = "Authentication failed", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=401,
            error_code="AUTHENTICATION_ERROR",
            details=details
        )


class PermissionError(BaseAppException):
    """Permission denied exception"""
    
    def __init__(self, message: str = "Permission denied", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=403,
            error_code="PERMISSION_DENIED",
            details=details
        )


class ConflictError(BaseAppException):
    """Resource conflict exception"""
    
    def __init__(self, message: str = "Resource conflict", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=409,
            error_code="CONFLICT",
            details=details
        )


class RateLimitError(BaseAppException):
    """Rate limit exceeded exception"""
    
    def __init__(self, message: str = "Rate limit exceeded", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=429,
            error_code="RATE_LIMIT_EXCEEDED",
            details=details
        )


class PaymentError(BaseAppException):
    """Payment processing error"""
    
    def __init__(self, message: str = "Payment error", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=402,
            error_code="PAYMENT_ERROR",
            details=details
        )


class RechargeError(BaseAppException):
    """Recharge related error"""
    
    def __init__(self, message: str = "Recharge error", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=400,
            error_code="RECHARGE_ERROR",
            details=details
        )


class KeyGenerationError(BaseAppException):
    """Key generation error"""
    
    def __init__(self, message: str = "Key generation failed", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=500,
            error_code="KEY_GENERATION_ERROR",
            details=details
        )


class BroadcastError(BaseAppException):
    """Broadcast delivery error"""
    
    def __init__(self, message: str = "Broadcast failed", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=500,
            error_code="BROADCAST_ERROR",
            details=details
        )


class DatabaseError(BaseAppException):
    """Database operation error"""
    
    def __init__(self, message: str = "Database error", details: Optional[Dict] = None):
        super().__init__(
            message=message,
            status_code=500,
            error_code="DATABASE_ERROR",
            details=details
        )