# backend/core/models.py
from sqlalchemy import (
    Column, Integer, String, BigInteger, Boolean, DateTime, 
    ForeignKey, Text, JSON, DECIMAL, Index, UniqueConstraint,
    CheckConstraint, TIMESTAMP, INET
)
from sqlalchemy.sql import func
from sqlalchemy.orm import relationship, validates
from datetime import datetime
import re

from .database import Base


class Owner(Base):
    """Owner/Super Admin model"""
    __tablename__ = "owners"

    id = Column(Integer, primary_key=True)
    telegram_id = Column(BigInteger, unique=True, nullable=False)
    username = Column(String(100), unique=True, nullable=False)
    first_name = Column(String(255))
    last_name = Column(String(255))
    email = Column(String(255))
    password_hash = Column(String(255), nullable=False)
    two_factor_enabled = Column(Boolean, default=False)
    two_factor_secret = Column(String(255))
    notification_preferences = Column(JSON, default={})
    language = Column(String(10), default="en")
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    last_login = Column(TIMESTAMP(timezone=True))
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())

    # Relationships
    admins = relationship("Admin", back_populates="owner", cascade="all, delete-orphan")
    recharge_history = relationship("RechargeHistory", back_populates="owner")
    audit_logs = relationship("AuditLog", foreign_keys="[AuditLog.actor_id]")

    @validates('telegram_id')
    def validate_telegram_id(self, key, value):
        if value <= 0:
            raise ValueError("Invalid telegram ID")
        return value

    @validates('email')
    def validate_email(self, key, value):
        if value and not re.match(r"[^@]+@[^@]+\.[^@]+", value):
            raise ValueError("Invalid email format")
        return value


class Admin(Base):
    """Admin model (per bot)"""
    __tablename__ = "admins"

    id = Column(Integer, primary_key=True)
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"), nullable=False)
    telegram_id = Column(BigInteger, unique=True, nullable=False)
    admin_username = Column(String(100), unique=True, nullable=False)
    admin_first_name = Column(String(255))
    admin_last_name = Column(String(255))
    email = Column(String(255))
    password_hash = Column(String(255), nullable=False)
    reserve_id = Column(String(50), unique=True, nullable=False)
    bot_name = Column(String(100), nullable=False)
    bot_username = Column(String(100), unique=True, nullable=False)
    bot_token = Column(String(255), unique=True, nullable=False)
    server_name = Column(String(255))
    server_ip = Column(String(50))
    server_port = Column(Integer, default=443)
    domain_name = Column(String(255))
    database_name = Column(String(100), unique=True)
    database_host = Column(String(255), default="localhost")
    database_port = Column(Integer, default=5432)
    database_user = Column(String(100))
    database_password = Column(Text)
    payment_gateway = Column(String(50), default="razorpay")
    payment_api_key = Column(Text)
    payment_api_secret = Column(Text)
    payment_merchant_id = Column(String(255))
    payment_webhook_secret = Column(Text)
    payment_currency = Column(String(10), default="INR")
    recharge_status = Column(String(20), default="inactive")
    recharge_plan = Column(String(20))
    recharge_amount = Column(DECIMAL(10, 2))
    recharge_date = Column(TIMESTAMP(timezone=True))
    recharge_expiry = Column(TIMESTAMP(timezone=True))
    auto_renew = Column(Boolean, default=False)
    monthly_key_limit = Column(Integer, default=1000)
    keys_generated_this_month = Column(Integer, default=0)
    total_keys_generated = Column(Integer, default=0)
    status = Column(String(20), default="active")
    settings = Column(JSON, default={})
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    last_login = Column(TIMESTAMP(timezone=True))
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())

    # Relationships
    owner = relationship("Owner", back_populates="admins")
    login_history = relationship("AdminLoginHistory", back_populates="admin", cascade="all, delete-orphan")
    activity_logs = relationship("AdminActivityLog", back_populates="admin", cascade="all, delete-orphan")
    tenant_registry = relationship("TenantRegistry", back_populates="admin", uselist=False, cascade="all, delete-orphan")
    server_health_logs = relationship("ServerHealthLog", back_populates="admin", cascade="all, delete-orphan")
    recharge_records = relationship("RechargeHistory", back_populates="admin")
    api_keys = relationship("APIKey", back_populates="admin")

    __table_args__ = (
        CheckConstraint(
            reserve_id ~ '^[A-Z0-9]{3,10}-[0-9]{3,10}$',
            name='valid_reserve_id'
        ),
        Index('idx_admins_recharge', 'recharge_expiry'),
    )

    @validates('bot_token')
    def validate_bot_token(self, key, value):
        if not re.match(r'^\d+:[A-Za-z0-9_-]+$', value):
            raise ValueError("Invalid bot token format")
        return value


class AdminLoginHistory(Base):
    __tablename__ = "admin_login_history"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"), nullable=False)
    login_time = Column(TIMESTAMP(timezone=True), server_default=func.now())
    ip_address = Column(INET)
    user_agent = Column(Text)
    login_successful = Column(Boolean, default=True)
    failure_reason = Column(Text)

    admin = relationship("Admin", back_populates="login_history")


class AdminActivityLog(Base):
    __tablename__ = "admin_activity_log"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"), nullable=False)
    reserve_id = Column(String(50), nullable=False)
    activity_type = Column(String(50), nullable=False)
    activity_details = Column(JSON, nullable=False)
    ip_address = Column(INET)
    performed_at = Column(TIMESTAMP(timezone=True), server_default=func.now())

    admin = relationship("Admin", back_populates="activity_logs")

    __table_args__ = (
        Index('idx_admin_activity_time', 'performed_at'),
    )


class TenantRegistry(Base):
    __tablename__ = "tenant_registry"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"), nullable=False, unique=True)
    reserve_id = Column(String(50), unique=True, nullable=False)
    database_name = Column(String(100), nullable=False)
    database_host = Column(String(255), default="localhost")
    database_port = Column(Integer, default=5432)
    database_user = Column(String(100))
    database_password = Column(Text)
    users_table = Column(String(100), nullable=False)
    keys_table = Column(String(100), nullable=False)
    transactions_table = Column(String(100), nullable=False)
    key_rules_table = Column(String(100), nullable=False)
    key_batches_table = Column(String(100), nullable=False)
    apk_versions_table = Column(String(100), nullable=False)
    broadcasts_table = Column(String(100), nullable=False)
    user_activity_table = Column(String(100), nullable=False)
    user_sessions_table = Column(String(100), nullable=False)
    referrals_table = Column(String(100), nullable=False)
    tickets_table = Column(String(100), nullable=False)
    key_alert_config_table = Column(String(100), nullable=False)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())

    admin = relationship("Admin", back_populates="tenant_registry")


class ServerHealthLog(Base):
    __tablename__ = "server_health_logs"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"), nullable=False)
    reserve_id = Column(String(50), nullable=False)
    server_name = Column(String(255))
    cpu_usage = Column(DECIMAL(5, 2))
    ram_usage = Column(DECIMAL(5, 2))
    disk_usage = Column(DECIMAL(5, 2))
    network_in = Column(DECIMAL(10, 2))
    network_out = Column(DECIMAL(10, 2))
    bot_status = Column(String(20))
    webhook_status = Column(String(20))
    database_status = Column(String(20))
    redis_status = Column(String(20))
    response_time = Column(Integer)
    checked_at = Column(TIMESTAMP(timezone=True), server_default=func.now())

    admin = relationship("Admin", back_populates="server_health_logs")


class ServerAlertRule(Base):
    __tablename__ = "server_alert_rules"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"))
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"))
    rule_name = Column(String(100), nullable=False)
    metric = Column(String(50), nullable=False)
    threshold = Column(DECIMAL(5, 2), nullable=False)
    operator = Column(String(10), nullable=False)
    duration = Column(Integer, default=5)
    notify_channels = Column(JSON, default=["telegram"])
    notify_owner = Column(Boolean, default=False)
    notify_admin = Column(Boolean, default=True)
    auto_restart = Column(Boolean, default=False)
    auto_scale = Column(Boolean, default=False)
    is_active = Column(Boolean, default=True)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())


class ServerIncident(Base):
    __tablename__ = "server_incidents"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"))
    reserve_id = Column(String(50), nullable=False)
    incident_type = Column(String(50), nullable=False)
    severity = Column(String(20), nullable=False)
    start_time = Column(TIMESTAMP(timezone=True), nullable=False)
    end_time = Column(TIMESTAMP(timezone=True))
    duration = Column(Integer)
    affected_users = Column(Integer)
    root_cause = Column(Text)
    resolution = Column(Text)
    status = Column(String(20), default="open")
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    resolved_at = Column(TIMESTAMP(timezone=True))


class RechargePlan(Base):
    __tablename__ = "recharge_plans"

    id = Column(Integer, primary_key=True)
    plan_name = Column(String(50), unique=True, nullable=False)
    duration_days = Column(Integer, nullable=False)
    price = Column(DECIMAL(10, 2), nullable=False)
    key_limit = Column(Integer, default=1000)
    description = Column(Text)
    is_active = Column(Boolean, default=True)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())


class RechargeHistory(Base):
    __tablename__ = "recharge_history"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"), nullable=False)
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"), nullable=False)
    plan_type = Column(String(20), nullable=False)
    amount = Column(DECIMAL(10, 2), nullable=False)
    payment_method = Column(String(50))
    transaction_id = Column(String(255))
    start_date = Column(TIMESTAMP(timezone=True), nullable=False)
    end_date = Column(TIMESTAMP(timezone=True), nullable=False)
    status = Column(String(20), default="completed")
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())

    admin = relationship("Admin", back_populates="recharge_records")
    owner = relationship("Owner", back_populates="recharge_history")


class SystemConfig(Base):
    __tablename__ = "system_config"

    id = Column(Integer, primary_key=True)
    config_key = Column(String(100), unique=True, nullable=False)
    config_value = Column(Text)
    config_type = Column(String(50), default="string")
    description = Column(Text)
    updated_by = Column(Integer)
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())


class AuditLog(Base):
    __tablename__ = "audit_logs"

    id = Column(Integer, primary_key=True)
    actor_type = Column(String(20), nullable=False)
    actor_id = Column(Integer)
    actor_telegram_id = Column(BigInteger)
    action = Column(String(100), nullable=False)
    entity_type = Column(String(50))
    entity_id = Column(String(100))
    old_values = Column(JSON)
    new_values = Column(JSON)
    ip_address = Column(INET)
    user_agent = Column(Text)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())


class BackupHistory(Base):
    __tablename__ = "backup_history"

    id = Column(Integer, primary_key=True)
    backup_id = Column(String(100), unique=True, nullable=False)
    backup_type = Column(String(50), nullable=False)
    backup_size = Column(BigInteger)
    file_path = Column(String(500))
    included_tables = Column(JSON)
    status = Column(String(20), default="completed")
    started_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    completed_at = Column(TIMESTAMP(timezone=True))
    error_message = Column(Text)
    created_by = Column(Integer)
    restored_at = Column(TIMESTAMP(timezone=True))
    restored_by = Column(Integer)


class APIKey(Base):
    __tablename__ = "api_keys"

    id = Column(Integer, primary_key=True)
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"))
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"))
    api_key = Column(String(255), unique=True, nullable=False)
    api_secret = Column(String(255))
    api_name = Column(String(100))
    permissions = Column(JSON, default=[])
    rate_limit = Column(Integer, default=100)
    last_used = Column(TIMESTAMP(timezone=True))
    expires_at = Column(TIMESTAMP(timezone=True))
    is_active = Column(Boolean, default=True)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())

    admin = relationship("Admin", back_populates="api_keys")
    owner = relationship("Owner")


class GlobalBroadcast(Base):
    __tablename__ = "broadcasts_global"

    id = Column(Integer, primary_key=True)
    broadcast_id = Column(String(50), unique=True, nullable=False)
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"))
    message = Column(Text, nullable=False)
    target_bots = Column(JSON, default=[])
    scheduled_for = Column(TIMESTAMP(timezone=True))
    sent_at = Column(TIMESTAMP(timezone=True))
    status = Column(String(20))
    total_recipients = Column(Integer)
    delivered_count = Column(Integer)
    failed_count = Column(Integer)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())


class BlockedUserGlobal(Base):
    __tablename__ = "blocked_users_global"

    id = Column(Integer, primary_key=True)
    telegram_id = Column(BigInteger, nullable=False)
    block_type = Column(String(20), default="global")
    reserve_id = Column(String(50))
    blocked_by = Column(Integer)
    block_reason = Column(Text)
    blocked_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    expires_at = Column(TIMESTAMP(timezone=True))

    __table_args__ = (
        UniqueConstraint('telegram_id', 'reserve_id', name='unique_block'),
    )


class NotificationTemplate(Base):
    __tablename__ = "notification_templates"

    id = Column(Integer, primary_key=True)
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"))
    template_name = Column(String(100), nullable=False)
    template_type = Column(String(50), nullable=False)
    subject = Column(Text)
    message = Column(Text, nullable=False)
    variables = Column(JSON, default=[])
    channels = Column(JSON, default=["telegram"])
    is_active = Column(Boolean, default=True)
    created_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    updated_at = Column(TIMESTAMP(timezone=True), onupdate=func.now())


class NotificationLog(Base):
    __tablename__ = "notification_logs"

    id = Column(Integer, primary_key=True)
    admin_id = Column(Integer, ForeignKey("admins.id", ondelete="CASCADE"))
    owner_id = Column(Integer, ForeignKey("owners.id", ondelete="CASCADE"))
    notification_type = Column(String(50), nullable=False)
    channel = Column(String(20), nullable=False)
    recipient_type = Column(String(20))
    recipient_id = Column(BigInteger)
    subject = Column(Text)
    message = Column(Text, nullable=False)
    status = Column(String(20), default="sent")
    error_message = Column(Text)
    sent_at = Column(TIMESTAMP(timezone=True), server_default=func.now())
    delivered_at = Column(TIMESTAMP(timezone=True))
    metadata = Column(JSON, default={})