# backend/core/security.py
from datetime import datetime, timedelta
from typing import Optional, Dict, Any, Union
from jose import JWTError, jwt
from passlib.context import CryptContext
from fastapi import HTTPException, status, Depends, Request
from fastapi.security import OAuth2PasswordBearer
import secrets
import hashlib
import hmac
import logging

from config.settings import settings
from core.database import get_db
from core.models import Owner, Admin

logger = logging.getLogger(__name__)

# Password hashing context
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

# OAuth2 scheme for token authentication
oauth2_scheme = OAuth2PasswordBearer(tokenUrl="/api/v1/auth/login")

class SecurityManager:
    """Security manager for authentication and authorization"""
    
    @staticmethod
    def verify_password(plain_password: str, hashed_password: str) -> bool:
        """Verify a plain password against a hash"""
        return pwd_context.verify(plain_password, hashed_password)
    
    @staticmethod
    def get_password_hash(password: str) -> str:
        """Hash a password"""
        return pwd_context.hash(password)
    
    @staticmethod
    def create_access_token(data: Dict[str, Any], expires_delta: Optional[timedelta] = None) -> str:
        """Create JWT access token"""
        to_encode = data.copy()
        if expires_delta:
            expire = datetime.utcnow() + expires_delta
        else:
            expire = datetime.utcnow() + timedelta(minutes=settings.JWT_ACCESS_TOKEN_EXPIRE_MINUTES)
        
        to_encode.update({
            "exp": expire,
            "type": "access",
            "iat": datetime.utcnow()
        })
        encoded_jwt = jwt.encode(to_encode, settings.JWT_SECRET_KEY, algorithm=settings.JWT_ALGORITHM)
        return encoded_jwt
    
    @staticmethod
    def create_refresh_token(data: Dict[str, Any]) -> str:
        """Create JWT refresh token"""
        to_encode = data.copy()
        expire = datetime.utcnow() + timedelta(days=settings.JWT_REFRESH_TOKEN_EXPIRE_DAYS)
        to_encode.update({
            "exp": expire,
            "type": "refresh",
            "iat": datetime.utcnow()
        })
        encoded_jwt = jwt.encode(to_encode, settings.JWT_SECRET_KEY, algorithm=settings.JWT_ALGORITHM)
        return encoded_jwt
    
    @staticmethod
    def decode_token(token: str) -> Optional[Dict[str, Any]]:
        """Decode JWT token"""
        try:
            payload = jwt.decode(token, settings.JWT_SECRET_KEY, algorithms=[settings.JWT_ALGORITHM])
            return payload
        except JWTError as e:
            logger.error(f"Token decode error: {e}")
            return None
    
    @staticmethod
    def generate_api_key() -> str:
        """Generate random API key"""
        return f"tk_{secrets.token_urlsafe(32)}"
    
    @staticmethod
    def generate_api_secret() -> str:
        """Generate random API secret"""
        return secrets.token_urlsafe(48)
    
    @staticmethod
    def verify_webhook_signature(payload: bytes, signature: str, secret: str) -> bool:
        """Verify webhook signature"""
        expected = hmac.new(
            secret.encode('utf-8'),
            payload,
            hashlib.sha256
        ).hexdigest()
        return hmac.compare_digest(signature, expected)


class AuthManager:
    """Authentication manager"""
    
    def __init__(self):
        self.security = SecurityManager()
    
    async def authenticate_owner(self, username: str, password: str, db) -> Optional[Owner]:
        """Authenticate owner by username and password"""
        owner = db.query(Owner).filter(Owner.username == username).first()
        if not owner:
            return None
        if not self.security.verify_password(password, owner.password_hash):
            return None
        return owner
    
    async def authenticate_admin(self, username: str, password: str, db) -> Optional[Admin]:
        """Authenticate admin by username and password"""
        admin = db.query(Admin).filter(Admin.admin_username == username).first()
        if not admin:
            return None
        if not self.security.verify_password(password, admin.password_hash):
            return None
        if admin.status != 'active':
            raise HTTPException(status_code=403, detail="Admin account is not active")
        return admin
    
    async def get_current_user(self, token: str, db):
        """Get current user from token"""
        payload = self.security.decode_token(token)
        if not payload:
            raise HTTPException(
                status_code=status.HTTP_401_UNAUTHORIZED,
                detail="Invalid authentication credentials",
                headers={"WWW-Authenticate": "Bearer"},
            )
        
        user_id: int = payload.get("sub")
        user_type: str = payload.get("user_type")
        
        if not user_id or not user_type:
            raise HTTPException(status_code=401, detail="Invalid token")
        
        if user_type == "owner":
            user = db.query(Owner).filter(Owner.id == user_id).first()
        elif user_type == "admin":
            user = db.query(Admin).filter(Admin.id == user_id).first()
        else:
            raise HTTPException(status_code=401, detail="Invalid user type")
        
        if not user:
            raise HTTPException(status_code=401, detail="User not found")
        
        return user


# Global instances
security = SecurityManager()
auth_manager = AuthManager()


async def get_current_user(
    token: str = Depends(oauth2_scheme),
    db = Depends(get_db)
):
    """Dependency to get current authenticated user"""
    return await auth_manager.get_current_user(token, db)


async def get_current_owner(current_user = Depends(get_current_user)):
    """Dependency to get current owner"""
    if not isinstance(current_user, Owner):
        raise HTTPException(status_code=403, detail="Not authorized as owner")
    return current_user


async def get_current_admin(current_user = Depends(get_current_user)):
    """Dependency to get current admin"""
    if not isinstance(current_user, Admin):
        raise HTTPException(status_code=403, detail="Not authorized as admin")
    return current_user


def verify_telegram_auth(telegram_data: Dict[str, str], bot_token: str) -> bool:
    """Verify Telegram authentication data"""
    # Implementation for Telegram login widget verification
    secret_key = hashlib.sha256(bot_token.encode()).digest()
    
    # Sort and concatenate data
    data_check = []
    for key in sorted(telegram_data.keys()):
        if key != 'hash':
            data_check.append(f"{key}={telegram_data[key]}")
    
    data_string = "\n".join(data_check)
    
    # Calculate hash
    hmac_hash = hmac.new(secret_key, data_string.encode(), hashlib.sha256).hexdigest()
    
    return hmac_hash == telegram_data.get('hash', '')