# backend/main.py
import logging
from fastapi import FastAPI, Request, Depends, HTTPException
from fastapi.middleware.cors import CORSMiddleware
from fastapi.responses import JSONResponse
from contextlib import asynccontextmanager
from typing import Optional
import time
import uvicorn

from config.settings import settings
from core.database import engine, check_database_health, get_redis_client
from core.utils import datetime_utils
from api.v1 import (
    auth, owner, admin, user, 
    keys, payments, broadcasts,
    activity, monitoring, system
)
from middleware import (
    RateLimitMiddleware, 
    LoggingMiddleware,
    AuthMiddleware
)
from tasks.celery_app import celery_app

# Configure logging
logging.basicConfig(
    level=getattr(logging, settings.LOG_LEVEL),
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)


@asynccontextmanager
async def lifespan(app: FastAPI):
    """
    Lifespan context manager for startup and shutdown events
    """
    # Startup
    logger.info("Starting up Telegram Multi-Tenant Bot System...")
    
    # Check database connection
    if not await check_database_health():
        logger.error("Database health check failed!")
        raise Exception("Database connection failed")
    
    # Initialize Redis
    app.state.redis = await get_redis_client()
    logger.info("Redis connection established")
    
    # Initialize background tasks
    from tasks import scheduler
    await scheduler.start_scheduler()
    
    logger.info(f"System started successfully in {settings.ENVIRONMENT} mode")
    
    yield
    
    # Shutdown
    logger.info("Shutting down...")
    
    # Close Redis connection
    await app.state.redis.close()
    
    # Stop scheduler
    await scheduler.stop_scheduler()
    
    # Dispose database connections
    await engine.dispose()
    
    logger.info("Shutdown complete")


# Create FastAPI app
app = FastAPI(
    title=settings.APP_NAME,
    version=settings.APP_VERSION,
    description="Enterprise Multi-Tenant Telegram Bot System",
    lifespan=lifespan,
    docs_url="/api/docs" if settings.DEBUG else None,
    redoc_url="/api/redoc" if settings.DEBUG else None,
)

# Add CORS middleware
app.add_middleware(
    CORSMiddleware,
    allow_origins=settings.CORS_ORIGINS,
    allow_credentials=settings.CORS_ALLOW_CREDENTIALS,
    allow_methods=settings.CORS_ALLOW_METHODS,
    allow_headers=settings.CORS_ALLOW_HEADERS,
)

# Add custom middleware
app.add_middleware(LoggingMiddleware)
if settings.RATE_LIMIT_ENABLED:
    app.add_middleware(RateLimitMiddleware)
app.add_middleware(AuthMiddleware)


# Request timing middleware
@app.middleware("http")
async def add_process_time_header(request: Request, call_next):
    start_time = time.time()
    response = await call_next(request)
    process_time = time.time() - start_time
    response.headers["X-Process-Time"] = str(process_time)
    return response


# Health check endpoint
@app.get("/health")
async def health_check():
    """
    Health check endpoint for monitoring
    """
    db_health = await check_database_health()
    redis_health = await app.state.redis.ping() if hasattr(app.state, 'redis') else False
    
    return {
        "status": "healthy" if db_health and redis_health else "unhealthy",
        "timestamp": datetime_utils.now().isoformat(),
        "environment": settings.ENVIRONMENT,
        "version": settings.APP_VERSION,
        "checks": {
            "database": "up" if db_health else "down",
            "redis": "up" if redis_health else "down",
        }
    }


# Root endpoint
@app.get("/")
async def root():
    return {
        "name": settings.APP_NAME,
        "version": settings.APP_VERSION,
        "environment": settings.ENVIRONMENT,
        "status": "operational"
    }


# Include API routers
app.include_router(auth.router, prefix="/api/v1/auth", tags=["Authentication"])
app.include_router(owner.router, prefix="/api/v1/owner", tags=["Owner"])
app.include_router(admin.router, prefix="/api/v1/admin", tags=["Admin"])
app.include_router(user.router, prefix="/api/v1/user", tags=["User"])
app.include_router(keys.router, prefix="/api/v1/keys", tags=["Keys"])
app.include_router(payments.router, prefix="/api/v1/payments", tags=["Payments"])
app.include_router(broadcasts.router, prefix="/api/v1/broadcasts", tags=["Broadcasts"])
app.include_router(activity.router, prefix="/api/v1/activity", tags=["Activity"])
app.include_router(monitoring.router, prefix="/api/v1/monitoring", tags=["Monitoring"])
app.include_router(system.router, prefix="/api/v1/system", tags=["System"])


# Error handlers
@app.exception_handler(HTTPException)
async def http_exception_handler(request: Request, exc: HTTPException):
    return JSONResponse(
        status_code=exc.status_code,
        content={
            "error": True,
            "message": exc.detail,
            "timestamp": datetime_utils.now().isoformat()
        }
    )


@app.exception_handler(Exception)
async def general_exception_handler(request: Request, exc: Exception):
    logger.error(f"Unhandled exception: {exc}", exc_info=True)
    return JSONResponse(
        status_code=500,
        content={
            "error": True,
            "message": "Internal server error",
            "timestamp": datetime_utils.now().isoformat()
        }
    )


if __name__ == "__main__":
    uvicorn.run(
        "main:app",
        host="0.0.0.0",
        port=8000,
        reload=settings.DEBUG,
        log_level=settings.LOG_LEVEL.lower()
    )