# backend/owner_panel/controllers/activity_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_, func
from datetime import datetime, date, timedelta
import logging
import csv
import io
from fastapi.responses import StreamingResponse, JSONResponse
from fastapi import Request
import asyncio

from core.models import Admin, AdminActivityLog, TenantRegistry
from core.exceptions import NotFoundError, ValidationError
from core.utils import datetime_utils
from admin_panel.services.tenant_service import TenantService
from shared.constants import ActivityType, AdminActivityType

logger = logging.getLogger(__name__)

class ActivityController:
    """Controller for activity tracking at owner level"""
    
    def __init__(self):
        self.tenant_service = TenantService()
        self.active_connections = []  # For SSE connections
    
    async def get_all_activities(self, filters: Dict[str, Any], skip: int, 
                                limit: int, db: Session) -> Dict[str, Any]:
        """Get all activities with filters"""
        activities = []
        total = 0
        
        # Get admin activities
        admin_query = db.query(AdminActivityLog)
        
        if filters.get('date_from'):
            admin_query = admin_query.filter(
                AdminActivityLog.performed_at >= filters['date_from']
            )
        if filters.get('date_to'):
            admin_query = admin_query.filter(
                AdminActivityLog.performed_at <= filters['date_to']
            )
        if filters.get('admin_id'):
            admin_query = admin_query.filter(
                AdminActivityLog.admin_id == filters['admin_id']
            )
        if filters.get('activity_type'):
            admin_query = admin_query.filter(
                AdminActivityLog.activity_type == filters['activity_type']
            )
        
        admin_activities = admin_query.order_by(
            AdminActivityLog.performed_at.desc()
        ).offset(skip).limit(limit).all()
        
        for act in admin_activities:
            activities.append({
                'type': 'admin',
                'id': act.id,
                'admin_id': act.admin_id,
                'reserve_id': act.reserve_id,
                'activity_type': act.activity_type,
                'details': act.activity_details,
                'ip_address': str(act.ip_address) if act.ip_address else None,
                'performed_at': act.performed_at
            })
        
        # Get user activities from all tenants
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            try:
                user_activities = await self.tenant_service.get_activities(
                    admin.reserve_id,
                    user_id=filters.get('user_id'),
                    activity_type=filters.get('user_activity_type'),
                    date_from=filters.get('date_from'),
                    date_to=filters.get('date_to'),
                    skip=0,
                    limit=limit // len(admins) if admins else limit
                )
                
                for act in user_activities:
                    activities.append({
                        'type': 'user',
                        'bot': admin.reserve_id,
                        'bot_name': admin.bot_name,
                        'user_id': act['user_id'],
                        'username': act.get('username'),
                        'activity_type': act['activity_type'],
                        'details': act['details'],
                        'ip_address': act.get('ip_address'),
                        'created_at': act['created_at']
                    })
                    
            except Exception as e:
                logger.error(f"Error fetching activities from admin {admin.id}: {e}")
                continue
        
        # Sort by timestamp
        activities.sort(key=lambda x: x.get('performed_at') or x.get('created_at'), reverse=True)
        
        return {
            'total': len(activities),
            'activities': activities[skip:skip+limit]
        }
    
    async def get_admin_activities(self, admin_id: Optional[int], date_from: Optional[date],
                                  date_to: Optional[date], activity_type: Optional[str],
                                  skip: int, limit: int, db: Session) -> Dict[str, Any]:
        """Get admin activities"""
        query = db.query(AdminActivityLog)
        
        if admin_id:
            query = query.filter(AdminActivityLog.admin_id == admin_id)
        if date_from:
            query = query.filter(AdminActivityLog.performed_at >= date_from)
        if date_to:
            date_to_end = datetime.combine(date_to, datetime.max.time())
            query = query.filter(AdminActivityLog.performed_at <= date_to_end)
        if activity_type:
            query = query.filter(AdminActivityLog.activity_type == activity_type)
        
        total = query.count()
        activities = query.order_by(
            AdminActivityLog.performed_at.desc()
        ).offset(skip).limit(limit).all()
        
        return {
            'total': total,
            'activities': [
                {
                    'id': a.id,
                    'admin_id': a.admin_id,
                    'reserve_id': a.reserve_id,
                    'activity_type': a.activity_type,
                    'details': a.activity_details,
                    'ip_address': str(a.ip_address) if a.ip_address else None,
                    'performed_at': a.performed_at
                }
                for a in activities
            ]
        }
    
    async def get_user_activities(self, telegram_id: Optional[int], bot_filter: Optional[str],
                                 date_from: Optional[date], date_to: Optional[date],
                                 activity_type: Optional[str], skip: int, limit: int,
                                 db: Session) -> Dict[str, Any]:
        """Get user activities across bots"""
        activities = []
        
        # Get all active admins
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            if bot_filter and admin.reserve_id != bot_filter:
                continue
            
            try:
                user_activities = await self.tenant_service.get_activities(
                    admin.reserve_id,
                    user_id=telegram_id,
                    activity_type=activity_type,
                    date_from=date_from,
                    date_to=date_to,
                    skip=0,
                    limit=limit
                )
                
                for act in user_activities:
                    activities.append({
                        'bot': admin.reserve_id,
                        'bot_name': admin.bot_name,
                        'user_id': act['user_id'],
                        'username': act.get('username'),
                        'activity_type': act['activity_type'],
                        'details': act['details'],
                        'ip_address': act.get('ip_address'),
                        'created_at': act['created_at']
                    })
                    
            except Exception as e:
                logger.error(f"Error fetching activities from admin {admin.id}: {e}")
                continue
        
        # Sort by timestamp
        activities.sort(key=lambda x: x['created_at'], reverse=True)
        
        return {
            'total': len(activities),
            'activities': activities[skip:skip+limit]
        }
    
    async def get_activity_stats(self, date_from: date, date_to: date, 
                                db: Session) -> Dict[str, Any]:
        """Get activity statistics"""
        stats = {
            'total_admin_activities': 0,
            'total_user_activities': 0,
            'by_type': {},
            'by_hour': {},
            'by_bot': {},
            'trends': []
        }
        
        # Admin activities stats
        admin_stats = db.query(
            AdminActivityLog.activity_type,
            func.count().label('count')
        ).filter(
            AdminActivityLog.performed_at.between(date_from, date_to)
        ).group_by(AdminActivityLog.activity_type).all()
        
        for stat in admin_stats:
            stats['by_type'][f"admin_{stat.activity_type}"] = stat.count
            stats['total_admin_activities'] += stat.count
        
        # User activities from all bots
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            try:
                user_stats = await self.tenant_service.get_activity_stats(
                    admin.reserve_id, date_from, date_to
                )
                
                stats['total_user_activities'] += user_stats.get('total', 0)
                stats['by_bot'][admin.reserve_id] = user_stats.get('total', 0)
                
                for act_type, count in user_stats.get('by_type', {}).items():
                    key = f"user_{act_type}"
                    stats['by_type'][key] = stats['by_type'].get(key, 0) + count
                
                for hour, count in user_stats.get('by_hour', {}).items():
                    stats['by_hour'][hour] = stats['by_hour'].get(hour, 0) + count
                
            except Exception as e:
                logger.error(f"Error fetching stats from admin {admin.id}: {e}")
                continue
        
        # Daily trends
        current = date_from
        while current <= date_to:
            next_day = current + timedelta(days=1)
            
            admin_count = db.query(AdminActivityLog).filter(
                AdminActivityLog.performed_at.between(current, next_day)
            ).count()
            
            user_count = 0
            for admin in admins:
                try:
                    count = await self.tenant_service.get_activity_count(
                        admin.reserve_id, current, next_day
                    )
                    user_count += count
                except:
                    continue
            
            stats['trends'].append({
                'date': current.isoformat(),
                'admin_activities': admin_count,
                'user_activities': user_count
            })
            
            current = next_day
        
        return stats
    
    async def export_activities(self, format: str, date_from: Optional[date],
                               date_to: Optional[date], bot_filter: Optional[str],
                               db: Session) -> StreamingResponse:
        """Export activities to CSV"""
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow([
            'Type', 'Bot', 'User/Admin ID', 'Username',
            'Activity Type', 'Details', 'IP Address', 'Timestamp'
        ])
        
        # Get admin activities
        admin_query = db.query(AdminActivityLog)
        if date_from:
            admin_query = admin_query.filter(AdminActivityLog.performed_at >= date_from)
        if date_to:
            date_to_end = datetime.combine(date_to, datetime.max.time())
            admin_query = admin_query.filter(AdminActivityLog.performed_at <= date_to_end)
        
        admin_activities = admin_query.all()
        
        for act in admin_activities:
            writer.writerow([
                'admin',
                act.reserve_id,
                act.admin_id,
                '',
                act.activity_type,
                str(act.activity_details),
                str(act.ip_address) if act.ip_address else '',
                act.performed_at.isoformat()
            ])
        
        # Get user activities
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            if bot_filter and admin.reserve_id != bot_filter:
                continue
            
            try:
                activities = await self.tenant_service.get_all_activities(
                    admin.reserve_id, date_from, date_to
                )
                
                for act in activities:
                    writer.writerow([
                        'user',
                        admin.reserve_id,
                        act['user_id'],
                        act.get('username', ''),
                        act['activity_type'],
                        str(act['details']),
                        act.get('ip_address', ''),
                        act['created_at'].isoformat()
                    ])
                    
            except Exception as e:
                logger.error(f"Error exporting activities from admin {admin.id}: {e}")
                continue
        
        output.seek(0)
        
        filename = f"activities_export_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
        
        return StreamingResponse(
            iter([output.getvalue()]),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={filename}"}
        )
    
    async def get_live_feed(self, request: Request, db: Session):
        """SSE endpoint for live activity feed"""
        async def event_generator():
            client_id = id(request.client)
            self.active_connections.append(client_id)
            
            try:
                while True:
                    # Check if client disconnected
                    if await request.is_disconnected():
                        break
                    
                    # Get recent activities
                    recent = await self.get_recent_activities(db)
                    
                    # Send as SSE
                    yield f"data: {recent}\n\n"
                    
                    await asyncio.sleep(5)  # Send every 5 seconds
                    
            finally:
                self.active_connections.remove(client_id)
        
        return JSONResponse(content={"type": "text/event-stream"})
    
    async def get_recent_activities(self, db: Session) -> List[Dict]:
        """Get recent activities for live feed"""
        activities = []
        
        # Get recent admin activities (last 30 seconds)
        cutoff = datetime_utils.now() - timedelta(seconds=30)
        admin_activities = db.query(AdminActivityLog).filter(
            AdminActivityLog.performed_at >= cutoff
        ).order_by(AdminActivityLog.performed_at.desc()).limit(10).all()
        
        for act in admin_activities:
            activities.append({
                'type': 'admin',
                'reserve_id': act.reserve_id,
                'activity': act.activity_type,
                'details': act.activity_details,
                'time': act.performed_at.isoformat()
            })
        
        # Get recent user activities
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            try:
                user_activities = await self.tenant_service.get_recent_activities(
                    admin.reserve_id, seconds=30, limit=5
                )
                
                for act in user_activities:
                    activities.append({
                        'type': 'user',
                        'bot': admin.reserve_id,
                        'user_id': act['user_id'],
                        'username': act.get('username'),
                        'activity': act['activity_type'],
                        'time': act['created_at'].isoformat()
                    })
                    
            except Exception as e:
                continue
        
        # Sort by time
        activities.sort(key=lambda x: x['time'], reverse=True)
        
        return activities[:20]