# backend/owner_panel/controllers/recharge_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_, func
from datetime import datetime, date, timedelta
import logging

from core.models import Admin, RechargeHistory, RechargePlan
from core.exceptions import NotFoundError, ValidationError, RechargeError
from core.utils import datetime_utils, id_generator
from shared.constants import RechargeStatus

logger = logging.getLogger(__name__)

class RechargeController:
    """Controller for recharge system management"""
    
    async def get_plans(self, db: Session) -> List[Dict[str, Any]]:
        """Get all recharge plans"""
        plans = db.query(RechargePlan).filter(
            RechargePlan.is_active == True
        ).all()
        
        return [
            {
                'id': p.id,
                'name': p.plan_name,
                'duration_days': p.duration_days,
                'price': float(p.price),
                'key_limit': p.key_limit,
                'description': p.description,
                'features': self._get_plan_features(p)
            }
            for p in plans
        ]
    
    async def create_plan(self, plan_data: Dict[str, Any], db: Session) -> Dict[str, Any]:
        """Create new recharge plan"""
        # Check if plan already exists
        existing = db.query(RechargePlan).filter(
            RechargePlan.plan_name == plan_data['plan_name']
        ).first()
        
        if existing:
            raise ValidationError(f"Plan {plan_data['plan_name']} already exists")
        
        plan = RechargePlan(
            plan_name=plan_data['plan_name'],
            duration_days=plan_data['duration_days'],
            price=plan_data['price'],
            key_limit=plan_data.get('key_limit', 1000),
            description=plan_data.get('description', ''),
            is_active=plan_data.get('is_active', True)
        )
        
        db.add(plan)
        db.commit()
        db.refresh(plan)
        
        logger.info(f"Recharge plan created: {plan.plan_name}")
        
        return {
            'id': plan.id,
            'name': plan.plan_name,
            'duration_days': plan.duration_days,
            'price': float(plan.price),
            'key_limit': plan.key_limit,
            'description': plan.description,
            'is_active': plan.is_active
        }
    
    async def recharge_admin(self, admin_id: int, recharge_data: Dict[str, Any], 
                            owner_id: int, db: Session) -> Dict[str, Any]:
        """Recharge admin panel"""
        admin = db.query(Admin).filter(Admin.id == admin_id).first()
        if not admin:
            raise NotFoundError(f"Admin {admin_id} not found")
        
        # Get plan details
        plan = db.query(RechargePlan).filter(
            RechargePlan.plan_name == recharge_data['plan_type']
        ).first()
        
        if not plan:
            raise ValidationError(f"Invalid plan: {recharge_data['plan_type']}")
        
        # Calculate dates
        start_date = datetime_utils.now()
        end_date = start_date + timedelta(days=plan.duration_days)
        
        # Create recharge record
        transaction_id = id_generator.generate_transaction_id()
        
        recharge = RechargeHistory(
            admin_id=admin_id,
            owner_id=owner_id,
            plan_type=recharge_data['plan_type'],
            amount=plan.price,
            payment_method=recharge_data.get('payment_method', 'manual'),
            transaction_id=transaction_id,
            start_date=start_date,
            end_date=end_date,
            status='completed'
        )
        
        db.add(recharge)
        
        # Update admin
        admin.recharge_status = 'active'
        admin.recharge_plan = plan.plan_name
        admin.recharge_amount = plan.price
        admin.recharge_date = start_date
        admin.recharge_expiry = end_date
        admin.monthly_key_limit = plan.key_limit
        admin.status = 'active'
        
        db.commit()
        
        logger.info(f"Admin {admin_id} recharged with plan {plan.plan_name}")
        
        return {
            'admin_id': admin_id,
            'admin_username': admin.admin_username,
            'plan': plan.plan_name,
            'amount': float(plan.price),
            'start_date': start_date.isoformat(),
            'end_date': end_date.isoformat(),
            'transaction_id': transaction_id,
            'key_limit': plan.key_limit
        }
    
    async def get_history(self, admin_id: Optional[int], date_from: Optional[date],
                         date_to: Optional[date], skip: int, limit: int,
                         db: Session) -> Dict[str, Any]:
        """Get recharge history"""
        query = db.query(RechargeHistory).join(Admin)
        
        if admin_id:
            query = query.filter(RechargeHistory.admin_id == admin_id)
        if date_from:
            query = query.filter(RechargeHistory.start_date >= date_from)
        if date_to:
            date_to_end = datetime.combine(date_to, datetime.max.time())
            query = query.filter(RechargeHistory.start_date <= date_to_end)
        
        total = query.count()
        history = query.order_by(
            RechargeHistory.created_at.desc()
        ).offset(skip).limit(limit).all()
        
        return {
            'total': total,
            'history': [
                {
                    'id': h.id,
                    'admin_id': h.admin_id,
                    'admin_username': h.admin.admin_username,
                    'bot_name': h.admin.bot_name,
                    'plan_type': h.plan_type,
                    'amount': float(h.amount),
                    'payment_method': h.payment_method,
                    'transaction_id': h.transaction_id,
                    'start_date': h.start_date.isoformat(),
                    'end_date': h.end_date.isoformat(),
                    'status': h.status,
                    'created_at': h.created_at.isoformat()
                }
                for h in history
            ]
        }
    
    async def get_expired_admins(self, db: Session) -> List[Dict[str, Any]]:
        """Get list of expired admins"""
        now = datetime_utils.now()
        
        expired = db.query(Admin).filter(
            Admin.recharge_expiry < now,
            Admin.recharge_status == 'active'
        ).all()
        
        expiring_soon = db.query(Admin).filter(
            Admin.recharge_expiry.between(now, now + timedelta(days=7)),
            Admin.recharge_status == 'active'
        ).all()
        
        return {
            'expired': [
                {
                    'id': a.id,
                    'username': a.admin_username,
                    'bot_name': a.bot_name,
                    'reserve_id': a.reserve_id,
                    'expiry_date': a.recharge_expiry.isoformat() if a.recharge_expiry else None
                }
                for a in expired
            ],
            'expiring_soon': [
                {
                    'id': a.id,
                    'username': a.admin_username,
                    'bot_name': a.bot_name,
                    'reserve_id': a.reserve_id,
                    'expiry_date': a.recharge_expiry.isoformat() if a.recharge_expiry else None,
                    'days_left': (a.recharge_expiry - now).days if a.recharge_expiry else 0
                }
                for a in expiring_soon
            ]
        }
    
    async def get_revenue_stats(self, period: str, db: Session) -> Dict[str, Any]:
        """Get revenue statistics from recharges"""
        now = datetime_utils.now()
        
        if period == 'today':
            start = now.replace(hour=0, minute=0, second=0, microsecond=0)
        elif period == 'week':
            start = now - timedelta(days=7)
        elif period == 'month':
            start = now - timedelta(days=30)
        elif period == 'year':
            start = now - timedelta(days=365)
        else:
            start = now - timedelta(days=30)
        
        recharges = db.query(RechargeHistory).filter(
            RechargeHistory.created_at >= start,
            RechargeHistory.status == 'completed'
        ).all()
        
        total = sum(r.amount for r in recharges)
        count = len(recharges)
        
        # Group by plan
        by_plan = {}
        for r in recharges:
            by_plan[r.plan_type] = by_plan.get(r.plan_type, 0) + float(r.amount)
        
        return {
            'period': period,
            'start_date': start.isoformat(),
            'end_date': now.isoformat(),
            'total_revenue': float(total),
            'total_transactions': count,
            'average_transaction': float(total / count) if count > 0 else 0,
            'by_plan': by_plan
        }
    
    def _get_plan_features(self, plan: RechargePlan) -> List[str]:
        """Get plan features list"""
        features = [
            f"{plan.key_limit} keys/month",
            "Full admin access",
            "Activity tracking"
        ]
        
        if plan.duration_days >= 180:
            features.append("Priority support")
        if plan.duration_days >= 365:
            features.append("Premium support")
            features.append("API access")
        
        return features