# backend/owner_panel/controllers/system_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import func
from datetime import datetime, timedelta
import logging
import platform
import psutil
import os

from core.models import Admin, Owner, SystemConfig, AuditLog, BackupHistory
from core.exceptions import NotFoundError, ValidationError
from core.utils import datetime_utils, id_generator
from shared.constants import AdminStatus, RechargeStatus

logger = logging.getLogger(__name__)

class SystemController:
    """Controller for system settings and management"""
    
    async def get_settings(self, db: Session) -> Dict[str, Any]:
        """Get system settings"""
        settings = db.query(SystemConfig).all()
        
        return {
            config.config_key: self._parse_config_value(config)
            for config in settings
        }
    
    async def update_settings(self, settings_update: Dict[str, Any], 
                             updated_by: int, db: Session) -> Dict[str, Any]:
        """Update system settings"""
        for key, value in settings_update.items():
            config = db.query(SystemConfig).filter(
                SystemConfig.config_key == key
            ).first()
            
            if config:
                config.config_value = str(value)
                config.updated_by = updated_by
                config.updated_at = datetime_utils.now()
            else:
                # Create new setting
                config = SystemConfig(
                    config_key=key,
                    config_value=str(value),
                    config_type=type(value).__name__,
                    updated_by=updated_by
                )
                db.add(config)
        
        db.commit()
        
        logger.info(f"System settings updated by user {updated_by}")
        
        return await self.get_settings(db)
    
    async def create_backup(self, created_by: int, db: Session) -> Dict[str, Any]:
        """Create system backup"""
        backup_id = id_generator.generate_batch_id()
        
        # In production, this would trigger actual backup process
        backup = BackupHistory(
            backup_id=backup_id,
            backup_type='manual',
            status='completed',
            started_at=datetime_utils.now(),
            completed_at=datetime_utils.now(),
            created_by=created_by
        )
        
        db.add(backup)
        db.commit()
        
        logger.info(f"System backup created: {backup_id}")
        
        return {
            'backup_id': backup_id,
            'status': 'completed',
            'created_at': backup.started_at.isoformat(),
            'size': backup.backup_size
        }
    
    async def get_audit_logs(self, skip: int, limit: int, db: Session) -> Dict[str, Any]:
        """Get audit logs"""
        total = db.query(AuditLog).count()
        logs = db.query(AuditLog).order_by(
            AuditLog.created_at.desc()
        ).offset(skip).limit(limit).all()
        
        return {
            'total': total,
            'logs': [
                {
                    'id': log.id,
                    'actor_type': log.actor_type,
                    'actor_id': log.actor_id,
                    'actor_telegram_id': log.actor_telegram_id,
                    'action': log.action,
                    'entity_type': log.entity_type,
                    'entity_id': log.entity_id,
                    'old_values': log.old_values,
                    'new_values': log.new_values,
                    'ip_address': str(log.ip_address) if log.ip_address else None,
                    'created_at': log.created_at.isoformat()
                }
                for log in logs
            ]
        }
    
    async def get_dashboard_stats(self, db: Session) -> Dict[str, Any]:
        """Get owner dashboard statistics"""
        now = datetime_utils.now()
        today_start = now.replace(hour=0, minute=0, second=0, microsecond=0)
        week_start = now - timedelta(days=7)
        month_start = now - timedelta(days=30)
        
        # Admin stats
        total_admins = db.query(Admin).count()
        active_admins = db.query(Admin).filter(
            Admin.status == 'active'
        ).count()
        expired_admins = db.query(Admin).filter(
            Admin.recharge_status == 'expired'
        ).count()
        
        # Revenue stats (from recharge_history)
        from core.models import RechargeHistory
        total_revenue = db.query(func.sum(RechargeHistory.amount)).filter(
            RechargeHistory.status == 'completed'
        ).scalar() or 0
        
        today_revenue = db.query(func.sum(RechargeHistory.amount)).filter(
            RechargeHistory.created_at >= today_start,
            RechargeHistory.status == 'completed'
        ).scalar() or 0
        
        week_revenue = db.query(func.sum(RechargeHistory.amount)).filter(
            RechargeHistory.created_at >= week_start,
            RechargeHistory.status == 'completed'
        ).scalar() or 0
        
        month_revenue = db.query(func.sum(RechargeHistory.amount)).filter(
            RechargeHistory.created_at >= month_start,
            RechargeHistory.status == 'completed'
        ).scalar() or 0
        
        # System health
        system_health = await self._get_system_health()
        
        # Recent activities
        recent_activities = db.query(AuditLog).order_by(
            AuditLog.created_at.desc()
        ).limit(10).all()
        
        # Top admins by revenue
        top_admins = db.query(
            Admin.id, Admin.admin_username, Admin.bot_name,
            func.sum(RechargeHistory.amount).label('total_revenue')
        ).join(
            RechargeHistory, Admin.id == RechargeHistory.admin_id
        ).filter(
            RechargeHistory.status == 'completed'
        ).group_by(Admin.id).order_by(
            func.sum(RechargeHistory.amount).desc()
        ).limit(5).all()
        
        return {
            'totalAdmins': total_admins,
            'activeAdmins': active_admins,
            'expiredAdmins': expired_admins,
            'totalRevenue': float(total_revenue),
            'todayRevenue': float(today_revenue),
            'weekRevenue': float(week_revenue),
            'monthRevenue': float(month_revenue),
            'systemHealth': system_health,
            'recentActivities': [
                {
                    'id': a.id,
                    'actor': f"{a.actor_type}:{a.actor_id}",
                    'action': a.action,
                    'entity': f"{a.entity_type}:{a.entity_id}" if a.entity_type else None,
                    'time': a.created_at.isoformat()
                }
                for a in recent_activities
            ],
            'topAdmins': [
                {
                    'id': a.id,
                    'username': a.admin_username,
                    'bot_name': a.bot_name,
                    'revenue': float(a.total_revenue)
                }
                for a in top_admins
            ]
        }
    
    async def _get_system_health(self) -> Dict[str, Any]:
        """Get system health metrics"""
        try:
            # CPU usage
            cpu_percent = psutil.cpu_percent(interval=1)
            
            # Memory usage
            memory = psutil.virtual_memory()
            
            # Disk usage
            disk = psutil.disk_usage('/')
            
            # System info
            system = platform.uname()
            
            return {
                'status': 'healthy' if cpu_percent < 80 and memory.percent < 80 else 'degraded',
                'cpu': {
                    'usage': cpu_percent,
                    'cores': psutil.cpu_count()
                },
                'memory': {
                    'total': memory.total,
                    'available': memory.available,
                    'used': memory.used,
                    'percent': memory.percent
                },
                'disk': {
                    'total': disk.total,
                    'used': disk.used,
                    'free': disk.free,
                    'percent': disk.percent
                },
                'system': {
                    'hostname': system.node,
                    'platform': system.system,
                    'release': system.release,
                    'version': system.version
                },
                'uptime': self._get_uptime()
            }
        except Exception as e:
            logger.error(f"Error getting system health: {e}")
            return {
                'status': 'unknown',
                'error': str(e)
            }
    
    def _get_uptime(self) -> float:
        """Get system uptime in seconds"""
        try:
            with open('/proc/uptime', 'r') as f:
                uptime_seconds = float(f.readline().split()[0])
                return uptime_seconds
        except:
            return 0
    
    def _parse_config_value(self, config: SystemConfig) -> Any:
        """Parse config value based on type"""
        if config.config_type == 'boolean':
            return config.config_value.lower() == 'true'
        elif config.config_type == 'number':
            try:
                return float(config.config_value)
            except:
                return 0
        elif config.config_type == 'json':
            import json
            try:
                return json.loads(config.config_value)
            except:
                return {}
        else:
            return config.config_value