# backend/owner_panel/controllers/user_controller.py
from typing import List, Optional, Dict, Any
from sqlalchemy.orm import Session
from sqlalchemy import and_, or_, func
from datetime import datetime, date
import logging
import csv
import io
from fastapi.responses import StreamingResponse

from core.models import Admin, TenantRegistry, BlockedUserGlobal
from core.exceptions import NotFoundError, ValidationError
from core.utils import datetime_utils
from shared.constants import ActivityType
from admin_panel.services.tenant_service import TenantService

logger = logging.getLogger(__name__)

class UserController:
    """Controller for user management at owner level"""
    
    def __init__(self):
        self.tenant_service = TenantService()
    
    async def get_all_users(self, skip: int, limit: int, search: Optional[str], 
                           bot_filter: Optional[str], db: Session) -> Dict[str, Any]:
        """Get all users across all bots"""
        users = []
        total_users = 0
        
        # Get all active admins
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            if bot_filter and admin.reserve_id != bot_filter:
                continue
            
            try:
                # Get tenant connection
                tenant = db.query(TenantRegistry).filter(
                    TenantRegistry.admin_id == admin.id
                ).first()
                
                if not tenant:
                    continue
                
                # Get users from tenant database
                tenant_users = await self.tenant_service.get_users(
                    admin.reserve_id, search, skip, limit
                )
                
                for user in tenant_users:
                    users.append({
                        'telegram_id': user['telegram_id'],
                        'username': user['username'],
                        'first_name': user['first_name'],
                        'bot': admin.reserve_id,
                        'bot_name': admin.bot_name,
                        'joined_at': user['joined_at'],
                        'last_active': user['last_active'],
                        'total_purchases': user['total_purchases'],
                        'total_spent': user['total_spent'],
                        'status': user['status']
                    })
                
                total_users += len(tenant_users)
                
            except Exception as e:
                logger.error(f"Error fetching users for admin {admin.id}: {e}")
                continue
        
        return {
            'total': total_users,
            'users': users[skip:skip+limit]
        }
    
    async def get_user_details(self, telegram_id: int, db: Session) -> Dict[str, Any]:
        """Get complete user details by Telegram ID"""
        user_data = {
            'telegram_id': telegram_id,
            'bots': [],
            'global_block': None
        }
        
        # Check global block
        blocked = db.query(BlockedUserGlobal).filter(
            BlockedUserGlobal.telegram_id == telegram_id
        ).first()
        
        if blocked:
            user_data['global_block'] = {
                'reason': blocked.block_reason,
                'blocked_at': blocked.blocked_at,
                'expires_at': blocked.expires_at
            }
        
        # Get all admins
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            try:
                # Get tenant connection
                tenant = db.query(TenantRegistry).filter(
                    TenantRegistry.admin_id == admin.id
                ).first()
                
                if not tenant:
                    continue
                
                # Get user details from tenant
                user = await self.tenant_service.get_user(
                    admin.reserve_id, telegram_id
                )
                
                if user:
                    # Get user activity
                    activity = await self.tenant_service.get_user_activity(
                        admin.reserve_id, telegram_id, limit=50
                    )
                    
                    # Get user keys
                    keys = await self.tenant_service.get_user_keys(
                        admin.reserve_id, telegram_id
                    )
                    
                    user_data['bots'].append({
                        'reserve_id': admin.reserve_id,
                        'bot_name': admin.bot_name,
                        'user_details': user,
                        'recent_activity': activity,
                        'keys': keys
                    })
                    
            except Exception as e:
                logger.error(f"Error fetching user from admin {admin.id}: {e}")
                continue
        
        if not user_data['bots'] and not user_data['global_block']:
            raise NotFoundError(f"User {telegram_id} not found in any bot")
        
        return user_data
    
    async def block_user(self, telegram_id: int, reason: str, 
                        reserve_id: Optional[str], db: Session) -> Dict[str, Any]:
        """Block user globally or per bot"""
        if reserve_id:
            # Block in specific bot
            admin = db.query(Admin).filter(Admin.reserve_id == reserve_id).first()
            if not admin:
                raise NotFoundError(f"Bot {reserve_id} not found")
            
            # Block in tenant
            await self.tenant_service.block_user(reserve_id, telegram_id, reason)
            
            return {
                'message': f'User blocked in bot {reserve_id}',
                'telegram_id': telegram_id,
                'bot': reserve_id,
                'reason': reason
            }
        else:
            # Block globally
            blocked = BlockedUserGlobal(
                telegram_id=telegram_id,
                block_type='global',
                block_reason=reason,
                blocked_at=datetime_utils.now()
            )
            db.add(blocked)
            db.commit()
            
            # Block in all bots
            admins = db.query(Admin).filter(Admin.status == 'active').all()
            for admin in admins:
                try:
                    await self.tenant_service.block_user(
                        admin.reserve_id, telegram_id, reason
                    )
                except:
                    continue
            
            return {
                'message': 'User blocked globally',
                'telegram_id': telegram_id,
                'reason': reason
            }
    
    async def unblock_user(self, telegram_id: int, reserve_id: Optional[str], 
                          db: Session) -> Dict[str, Any]:
        """Unblock user"""
        if reserve_id:
            # Unblock in specific bot
            admin = db.query(Admin).filter(Admin.reserve_id == reserve_id).first()
            if not admin:
                raise NotFoundError(f"Bot {reserve_id} not found")
            
            await self.tenant_service.unblock_user(reserve_id, telegram_id)
            
            return {
                'message': f'User unblocked in bot {reserve_id}',
                'telegram_id': telegram_id,
                'bot': reserve_id
            }
        else:
            # Remove global block
            db.query(BlockedUserGlobal).filter(
                BlockedUserGlobal.telegram_id == telegram_id
            ).delete()
            db.commit()
            
            # Unblock in all bots
            admins = db.query(Admin).filter(Admin.status == 'active').all()
            for admin in admins:
                try:
                    await self.tenant_service.unblock_user(
                        admin.reserve_id, telegram_id
                    )
                except:
                    continue
            
            return {
                'message': 'User unblocked globally',
                'telegram_id': telegram_id
            }
    
    async def export_users(self, format: str, bot_filter: Optional[str], 
                          db: Session) -> StreamingResponse:
        """Export users to CSV"""
        # Create CSV in memory
        output = io.StringIO()
        writer = csv.writer(output)
        
        # Write header
        writer.writerow([
            'Telegram ID', 'Username', 'First Name', 'Last Name',
            'Bot', 'Joined Date', 'Last Active', 'Total Purchases',
            'Total Spent', 'Status'
        ])
        
        # Get all active admins
        admins = db.query(Admin).filter(Admin.status == 'active').all()
        
        for admin in admins:
            if bot_filter and admin.reserve_id != bot_filter:
                continue
            
            try:
                # Get users from tenant
                users = await self.tenant_service.get_all_users(admin.reserve_id)
                
                for user in users:
                    writer.writerow([
                        user['telegram_id'],
                        user.get('username', ''),
                        user.get('first_name', ''),
                        user.get('last_name', ''),
                        admin.reserve_id,
                        user.get('joined_at', ''),
                        user.get('last_active', ''),
                        user.get('total_purchases', 0),
                        user.get('total_spent', 0),
                        user.get('status', 'active')
                    ])
                    
            except Exception as e:
                logger.error(f"Error exporting users from admin {admin.id}: {e}")
                continue
        
        output.seek(0)
        
        filename = f"users_export_{datetime_utils.now().strftime('%Y%m%d_%H%M%S')}.csv"
        
        return StreamingResponse(
            iter([output.getvalue()]),
            media_type="text/csv",
            headers={"Content-Disposition": f"attachment; filename={filename}"}
        )