# backend/owner_panel/routes.py
from fastapi import APIRouter, Depends, HTTPException, Query
from typing import List, Optional
from datetime import datetime, date

from core.database import get_db
from core.security import get_current_owner
from shared.decorators.role_required import owner_required
from .controllers import (
    AdminController,
    UserController,
    ActivityController,
    RechargeController,
    SystemController
)
from .schemas import (
    AdminCreate, AdminResponse, AdminUpdate,
    UserResponse, UserListResponse,
    ActivityLogResponse, ActivityFilter,
    RechargePlanResponse, RechargeRequest,
    SystemSettingsResponse, SystemSettingsUpdate
)

router = APIRouter(prefix="/owner", tags=["Owner Panel"])

# ==================== Admin Management ====================

@router.post("/admins", response_model=AdminResponse)
@owner_required
async def create_admin(
    admin_data: AdminCreate,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Create new admin with complete configuration"""
    return await admin_controller.create_admin(admin_data, current_owner.id, db)

@router.get("/admins", response_model=List[AdminResponse])
@owner_required
async def get_all_admins(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    status: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Get all admins with optional filtering"""
    return await admin_controller.get_admins(skip, limit, status, db)

@router.get("/admins/{admin_id}", response_model=AdminResponse)
@owner_required
async def get_admin_details(
    admin_id: int,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Get detailed admin information"""
    return await admin_controller.get_admin(admin_id, db)

@router.put("/admins/{admin_id}", response_model=AdminResponse)
@owner_required
async def update_admin(
    admin_id: int,
    admin_data: AdminUpdate,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Update admin configuration"""
    return await admin_controller.update_admin(admin_id, admin_data, db)

@router.delete("/admins/{admin_id}")
@owner_required
async def delete_admin(
    admin_id: int,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Remove admin from system"""
    return await admin_controller.delete_admin(admin_id, db)

@router.post("/admins/{admin_id}/block")
@owner_required
async def block_admin(
    admin_id: int,
    reason: str = Query(...),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Block admin access"""
    return await admin_controller.block_admin(admin_id, reason, db)

@router.post("/admins/{admin_id}/unblock")
@owner_required
async def unblock_admin(
    admin_id: int,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    admin_controller: AdminController = Depends()
):
    """Unblock admin access"""
    return await admin_controller.unblock_admin(admin_id, db)

# ==================== User Management ====================

@router.get("/users", response_model=UserListResponse)
@owner_required
async def get_all_users(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    search: Optional[str] = None,
    bot_filter: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get all users across all bots"""
    return await user_controller.get_all_users(skip, limit, search, bot_filter, db)

@router.get("/users/{telegram_id}", response_model=UserResponse)
@owner_required
async def get_user_details(
    telegram_id: int,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Get complete user details by Telegram ID"""
    return await user_controller.get_user_details(telegram_id, db)

@router.post("/users/{telegram_id}/block")
@owner_required
async def block_user_global(
    telegram_id: int,
    reason: str = Query(...),
    reserve_id: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Block user globally or per bot"""
    return await user_controller.block_user(telegram_id, reason, reserve_id, db)

@router.post("/users/{telegram_id}/unblock")
@owner_required
async def unblock_user(
    telegram_id: int,
    reserve_id: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Unblock user"""
    return await user_controller.unblock_user(telegram_id, reserve_id, db)

@router.get("/users/export")
@owner_required
async def export_users(
    format: str = Query("csv", regex="^(csv|pdf)$"),
    bot_filter: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    user_controller: UserController = Depends()
):
    """Export all users to CSV or PDF"""
    return await user_controller.export_users(format, bot_filter, db)

# ==================== Activity Tracking ====================

@router.get("/activities", response_model=List[ActivityLogResponse])
@owner_required
async def get_all_activities(
    filters: ActivityFilter = Depends(),
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Get all admin and user activities with filters"""
    return await activity_controller.get_all_activities(filters, skip, limit, db)

@router.get("/activities/admin")
@owner_required
async def get_admin_activities(
    admin_id: Optional[int] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    activity_type: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Get admin activities with filters"""
    return await activity_controller.get_admin_activities(
        admin_id, date_from, date_to, activity_type, skip, limit, db
    )

@router.get("/activities/user")
@owner_required
async def get_user_activities(
    telegram_id: Optional[int] = None,
    bot_filter: Optional[str] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    activity_type: Optional[str] = None,
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Get user activities with filters"""
    return await activity_controller.get_user_activities(
        telegram_id, bot_filter, date_from, date_to, activity_type, skip, limit, db
    )

@router.get("/activities/stats")
@owner_required
async def get_activity_stats(
    date_from: date,
    date_to: date,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Get activity statistics"""
    return await activity_controller.get_activity_stats(date_from, date_to, db)

@router.get("/activities/export")
@owner_required
async def export_activities(
    format: str = Query("csv", regex="^(csv|pdf)$"),
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    bot_filter: Optional[str] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Export activity report"""
    return await activity_controller.export_activities(
        format, date_from, date_to, bot_filter, db
    )

@router.get("/activities/live")
@owner_required
async def get_live_activity_feed(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    activity_controller: ActivityController = Depends()
):
    """Get real-time activity feed (SSE)"""
    return await activity_controller.get_live_feed(db)

# ==================== Recharge System ====================

@router.get("/recharge/plans", response_model=List[RechargePlanResponse])
@owner_required
async def get_recharge_plans(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    recharge_controller: RechargeController = Depends()
):
    """Get all recharge plans"""
    return await recharge_controller.get_plans(db)

@router.post("/recharge/plans")
@owner_required
async def create_recharge_plan(
    plan_data: RechargePlanCreate,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    recharge_controller: RechargeController = Depends()
):
    """Create new recharge plan"""
    return await recharge_controller.create_plan(plan_data, db)

@router.post("/recharge/admin/{admin_id}")
@owner_required
async def recharge_admin(
    admin_id: int,
    recharge_data: RechargeRequest,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    recharge_controller: RechargeController = Depends()
):
    """Recharge admin panel"""
    return await recharge_controller.recharge_admin(admin_id, recharge_data, current_owner.id, db)

@router.get("/recharge/history")
@owner_required
async def get_recharge_history(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    admin_id: Optional[int] = None,
    date_from: Optional[date] = None,
    date_to: Optional[date] = None,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    recharge_controller: RechargeController = Depends()
):
    """Get recharge history"""
    return await recharge_controller.get_history(admin_id, date_from, date_to, skip, limit, db)

@router.get("/recharge/expired")
@owner_required
async def get_expired_admins(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    recharge_controller: RechargeController = Depends()
):
    """Get list of expired admins"""
    return await recharge_controller.get_expired_admins(db)

# ==================== System Settings ====================

@router.get("/settings", response_model=SystemSettingsResponse)
@owner_required
async def get_system_settings(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    system_controller: SystemController = Depends()
):
    """Get system settings"""
    return await system_controller.get_settings(db)

@router.put("/settings", response_model=SystemSettingsResponse)
@owner_required
async def update_system_settings(
    settings_update: SystemSettingsUpdate,
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    system_controller: SystemController = Depends()
):
    """Update system settings"""
    return await system_controller.update_settings(settings_update, current_owner.id, db)

@router.post("/settings/backup")
@owner_required
async def create_system_backup(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    system_controller: SystemController = Depends()
):
    """Create system backup"""
    return await system_controller.create_backup(current_owner.id, db)

@router.get("/settings/audit-logs")
@owner_required
async def get_audit_logs(
    skip: int = Query(0, ge=0),
    limit: int = Query(100, ge=1, le=1000),
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    system_controller: SystemController = Depends()
):
    """Get audit logs"""
    return await system_controller.get_audit_logs(skip, limit, db)

# ==================== Dashboard Stats ====================

@router.get("/dashboard/stats")
@owner_required
async def get_dashboard_stats(
    current_owner = Depends(get_current_owner),
    db = Depends(get_db),
    system_controller: SystemController = Depends()
):
    """Get owner dashboard statistics"""
    return await system_controller.get_dashboard_stats(db)