# backend/owner_panel/schemas/admin_schema.py
from pydantic import BaseModel, Field, validator
from typing import Optional, List, Dict, Any
from datetime import datetime

class AdminCreate(BaseModel):
    """Schema for creating admin"""
    telegram_id: int = Field(..., description="Admin's Telegram ID")
    admin_username: str = Field(..., min_length=3, max_length=100)
    password: str = Field(..., min_length=8)
    first_name: Optional[str] = Field(None)
    last_name: Optional[str] = Field(None)
    email: Optional[str] = Field(None)
    bot_name: str = Field(..., min_length=1, max_length=100)
    bot_username: str = Field(..., min_length=5, max_length=100)
    bot_token: str = Field(..., min_length=40, max_length=100)
    payment_gateway: str = Field("razorpay")
    payment_api_key: Optional[str] = Field(None)
    payment_api_secret: Optional[str] = Field(None)
    monthly_key_limit: int = Field(1000, ge=100, le=100000)
    settings: Optional[Dict[str, Any]] = Field(default_factory=dict)
    
    @validator('telegram_id')
    def validate_telegram_id(cls, v):
        if v <= 0:
            raise ValueError("Invalid Telegram ID")
        return v
    
    @validator('bot_token')
    def validate_bot_token(cls, v):
        if ':' not in v:
            raise ValueError("Invalid bot token format")
        return v
    
    @validator('email')
    def validate_email(cls, v):
        if v and '@' not in v:
            raise ValueError("Invalid email format")
        return v

class AdminUpdate(BaseModel):
    """Schema for updating admin"""
    first_name: Optional[str] = None
    last_name: Optional[str] = None
    email: Optional[str] = None
    password: Optional[str] = None
    payment_gateway: Optional[str] = None
    payment_api_key: Optional[str] = None
    payment_api_secret: Optional[str] = None
    monthly_key_limit: Optional[int] = None
    settings: Optional[Dict[str, Any]] = None

class AdminResponse(BaseModel):
    """Schema for admin response"""
    id: int
    telegram_id: int
    admin_username: str
    first_name: Optional[str]
    last_name: Optional[str]
    email: Optional[str]
    reserve_id: str
    bot_name: str
    bot_username: str
    payment_gateway: str
    payment_currency: str
    recharge_status: str
    recharge_plan: Optional[str]
    recharge_expiry: Optional[datetime]
    monthly_key_limit: int
    keys_generated_this_month: int
    status: str
    created_at: datetime
    last_login: Optional[datetime]
    
    class Config:
        from_attributes = True

class AdminListResponse(BaseModel):
    """Schema for list of admins response"""
    total: int
    admins: List[AdminResponse]

class AdminStatsResponse(BaseModel):
    """Schema for admin statistics"""
    admin_id: int
    username: str
    reserve_id: str
    bot_name: str
    status: str
    recharge: Dict[str, Any]
    tenant_stats: Dict[str, Any]
    created_at: str
    last_login: Optional[str]